package io.fincast.holding.impl.core

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProviders.constValue
import io.fincast.compo.impl.CashflowCompo
import io.fincast.compo.impl.FlowDirection
import io.fincast.compo.impl.InterestCompo
import io.fincast.enums.FundsUtilisation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.holding.base.ValuableBase
import io.fincast.household.Person

/**
 * A tangible Asset.
 *
 * @property tag tag (non-unique) to map back to client entity
 * @property owner the owner of the asset
 * @property reconBalance the balance of the asset at the reconDate
 * @property insuranceRate the insurance rate of the asset (if any)
 * @property insurancePeriodicity the periodicity of the insurance payments
 * @property maintenanceRate the maintenance rate of the asset (if any)
 * @property maintenancePeriodicity the periodicity of the maintenance payments
 * @property yieldRate the yield rate of the asset (if any)
 * @property yieldPeriodicity the periodicity of the yield payments
 * @property yieldUtilisation the use of the yield
 */
data class TangibleAsset(
	override val tag: String,
	override val owner: Person? = null,
	override val taxCode: String? = null,
	override val reconBalance: Double = 0.0,
	val insuranceRate: Double? = null,
	val insurancePeriodicity: Periodicity = Periodicity.YEARLY,
	val maintenanceRate: Double? = null,
	val maintenancePeriodicity: Periodicity = Periodicity.YEARLY,
	val yieldRate: Double? = null,
	val yieldPeriodicity: Periodicity = Periodicity.YEARLY,
	val yieldUtilisation: FundsUtilisation = FundsUtilisation.DISBURSE,
) : ValuableBase(tag, owner, ProductType.ASSET, taxCode, reconBalance) {

	class Builder {
		private var tag: String? = null
		private var taxCode: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var insuranceRate: Double? = null
		private var insurancePeriodicity: Periodicity? = null
		private var maintenanceRate: Double? = null
		private var maintenancePeriodicity: Periodicity? = null
		private var yieldRate: Double? = null
		private var yieldPeriodicity: Periodicity? = null
		private var yieldUtilisation: FundsUtilisation? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun taxCode(taxCode: String?) = apply { this.taxCode = taxCode }
		fun owner(owner: Person) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun insuranceRate(insuranceRate: Double?) = apply { this.insuranceRate = insuranceRate }
		fun insurancePeriodicity(insurancePeriodicity: Periodicity?) = apply { this.insurancePeriodicity = insurancePeriodicity }
		fun maintenanceRate(maintenanceRate: Double?) = apply { this.maintenanceRate = maintenanceRate }
		fun maintenancePeriodicity(maintenancePeriodicity: Periodicity?) = apply { this.maintenancePeriodicity = maintenancePeriodicity }
		fun yieldRate(yieldRate: Double?) = apply { this.yieldRate = yieldRate }
		fun yieldPeriodicity(yieldPeriodicity: Periodicity?) = apply { this.yieldPeriodicity = yieldPeriodicity }
		fun yieldUtilisation(yieldUtilisation: FundsUtilisation?) = apply { this.yieldUtilisation = yieldUtilisation }
		fun build(): TangibleAsset {
			return TangibleAsset(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				taxCode = taxCode,
				owner = owner,
				reconBalance = reconBalance ?: 0.0,
				insuranceRate = insuranceRate,
				insurancePeriodicity = insurancePeriodicity ?: Periodicity.YEARLY,
				maintenanceRate = maintenanceRate,
				maintenancePeriodicity = maintenancePeriodicity ?: Periodicity.YEARLY,
				yieldRate = yieldRate,
				yieldPeriodicity = yieldPeriodicity ?: Periodicity.YEARLY,
				yieldUtilisation = yieldUtilisation ?: FundsUtilisation.DISBURSE,
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		val compos: MutableList<FinancialCompo> = mutableListOf()
		if (insuranceRate != null && insuranceRate != 0.0) {
			compos.add(
				CashflowCompo(
					holding = this,
					tag = "insurance",
					fundsUtilisation = FundsUtilisation.DISBURSE,
					direction = FlowDirection.OUTGOING,
					amount = constValue(insuranceRate / 100.0 * reconBalance * insurancePeriodicity.months / 12),
					periodicity = insurancePeriodicity,
				)
			)
		}
		if (maintenanceRate != null && maintenanceRate != 0.0) {
			compos.add(
				CashflowCompo(
					holding = this,
					tag = "maintenance",
					fundsUtilisation = FundsUtilisation.DISBURSE,
					direction = FlowDirection.OUTGOING,
					amount = constValue(maintenanceRate / 100.0 * reconBalance * maintenancePeriodicity.months / 12),
					periodicity = maintenancePeriodicity,
				)
			)
		}
		if (yieldRate != null && yieldRate != 0.0) {
			compos.add(
				InterestCompo(
					holding = this,
					interestRate = constValue(yieldRate),
					interestPeriodicity = yieldPeriodicity,
					fundsUtilisation = yieldUtilisation,
				)
			)
		}
		return compos
	}

}
