package io.fincast.holding.impl.core

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProviders.constValue
import io.fincast.compo.impl.CashflowCompo
import io.fincast.compo.impl.FlowDirection
import io.fincast.enums.FundsUtilisation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.holding.base.ValuableBase
import io.fincast.household.Person

/**
 * A real estate asset.
 *
 * @property tag tag (non-unique) to map back to client entity
 * @property owner the owner of the asset
 * @property reconBalance the balance of the asset at the reconDate
 * @property fiscalValue the fiscal value of the asset
 * @property isSelfOccupied whether the real estate is self-occupied
 * @property imputedRentalValue the imputed rental value of the real estate, if self-occupied
 * @property rentalIncome the yearly rental income of the real estate, if rented out
 * @property maintenanceRate the maintenance rate of the real estate
 * @property maintenancePeriodicity the periodicity of the maintenance payments
 */
data class RealEstate(
	override val tag: String,
	override val owner: Person? = null,
	override val taxCode: String? = null,
	override val reconBalance: Double = 0.0, // market value
	val fiscalValue: Double? = null,
	val isSelfOccupied: Boolean? = null,
	val imputedRentalValue: Double? = null,
	val rentalIncome: Double? = null,
	val maintenanceRate: Double? = null,
	val maintenancePeriodicity: Periodicity = Periodicity.YEARLY,
) : ValuableBase(tag, owner, ProductType.ASSET, taxCode, reconBalance) {

	class Builder {
		private var tag: String? = null
		private var taxCode: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var fiscalValue: Double? = null
		private var isSelfOccupied: Boolean? = null
		private var imputedRentalValue: Double? = null
		private var rentalIncome: Double? = null
		private var maintenanceRate: Double? = null
		private var maintenancePeriodicity: Periodicity? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun taxCode(taxCode: String?) = apply { this.taxCode = taxCode }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun fiscalValue(fiscalValue: Double?) = apply { this.fiscalValue = fiscalValue }
		fun isSelfOccupied(isSelfOccupied: Boolean?) = apply { this.isSelfOccupied = isSelfOccupied }
		fun imputedRentalValue(imputedRentalValue: Double?) = apply { this.imputedRentalValue = imputedRentalValue }
		fun rentalIncome(rentalIncome: Double?) = apply { this.rentalIncome = rentalIncome }
		fun maintenanceRate(maintenanceRate: Double?) = apply { this.maintenanceRate = maintenanceRate }
		fun maintenancePeriodicity(maintenancePeriodicity: Periodicity?) = apply { this.maintenancePeriodicity = maintenancePeriodicity }
		fun build(): RealEstate {
			return RealEstate(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				taxCode = taxCode,
				owner = owner,
				reconBalance = reconBalance ?: 0.0,
				fiscalValue = fiscalValue,
				isSelfOccupied = isSelfOccupied,
				imputedRentalValue = imputedRentalValue,
				rentalIncome = rentalIncome,
				maintenanceRate = maintenanceRate,
				maintenancePeriodicity = maintenancePeriodicity ?: Periodicity.YEARLY,
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		val compos: MutableList<FinancialCompo> = mutableListOf()
		if (maintenanceRate != null && maintenanceRate != 0.0) {
			compos.add(
				CashflowCompo(
					holding = this,
					tag = "maintenance",
					fundsUtilisation = FundsUtilisation.DISBURSE,
					direction = FlowDirection.OUTGOING,
					amount = constValue(maintenanceRate / 100.0 * reconBalance * maintenancePeriodicity.months / 12),
					periodicity = maintenancePeriodicity,
				)
			)
		}
		if (rentalIncome != null && rentalIncome != 0.0) {
			compos.add(
				CashflowCompo(
					holding = this,
					tag = "rentalIncome",
					fundsUtilisation = FundsUtilisation.DISBURSE,
					direction = FlowDirection.INCOMING,
					amount = constValue(rentalIncome / 12),
					periodicity = Periodicity.MONTHLY,
				)
			)
		}
		return compos
	}

}
