package io.fincast.holding.impl.core

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProviders.constValue
import io.fincast.compo.impl.InterestCompo
import io.fincast.enums.FundsUtilisation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.holding.base.ValuableBase
import io.fincast.household.Person

/**
 * A generic liability.
 *
 * @property tag tag (non-unique) to map back to client entity
 * @property owner the owner of the liability
 * @property reconBalance the balance of the liability at the reconDate
 * @property interestRate the interest rate of the liability (if any)
 * @property interestPeriodicity the periodicity of the interest payments
 * @property interestUtilisation the use of the interest, i.e. whether it is reinvested or disbursed
 */
data class Liability(
	override val tag: String,
	override val owner: Person? = null,
	override val taxCode: String? = null,
	override val reconBalance: Double = 0.0,
	val interestRate: Double = 0.0,
	val interestPeriodicity: Periodicity = Periodicity.YEARLY,
	val interestUtilisation: FundsUtilisation = FundsUtilisation.DISBURSE,
) : ValuableBase(tag, owner, ProductType.LIABILITY, taxCode, reconBalance) {

	class Builder {
		private var tag: String? = null
		private var taxCode: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var interestRate: Double? = null
		private var interestPeriodicity: Periodicity? = null
		private var interestUtilisation: FundsUtilisation? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun taxCode(taxCode: String?) = apply { this.taxCode = taxCode }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun interestRate(interestRate: Double?) = apply { this.interestRate = interestRate }
		fun interestPeriodicity(interestPeriodicity: Periodicity?) = apply { this.interestPeriodicity = interestPeriodicity }
		fun interestUtilisation(interestUtilisation: FundsUtilisation?) = apply { this.interestUtilisation = interestUtilisation }
		fun build(): Liability {
			return Liability(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				taxCode = taxCode,
				owner = owner,
				reconBalance = reconBalance ?: 0.0,
				interestRate = interestRate ?: 0.0,
				interestPeriodicity = interestPeriodicity ?: Periodicity.YEARLY,
				interestUtilisation = interestUtilisation ?: FundsUtilisation.DISBURSE,
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		val compos: MutableList<FinancialCompo> = mutableListOf()
		if (interestRate != 0.0) {
			compos.add(
				InterestCompo(
					holding = this,
					interestRate = constValue(interestRate),
					interestPeriodicity = interestPeriodicity,
					fundsUtilisation = interestUtilisation,
				)
			)
		}
		return compos
	}

}
