package io.fincast.holding.impl.core

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProvider
import io.fincast.compo.ValueProviders.constValue
import io.fincast.compo.impl.*
import io.fincast.engine.SimDate
import io.fincast.enums.FundsUtilisation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.holding.base.ValuableBase
import io.fincast.household.Person

data class AssetAllocation(
	val cashPart: Double = 0.0,
	val cashInterestRate: Double? = null,
	val bondPart: Double = 0.0,
	val bondInterestRate: Double? = null,
	val stockPart: Double = 0.0,
	val stockCapitalGain: Double? = null,
	val stockDividendYield: Double? = null,
) {

	class Builder {
		private var cashPart: Double? = null
		private var cashInterestRate: Double? = null
		private var bondPart: Double? = null
		private var bondInterestRate: Double? = null
		private var stockPart: Double? = null
		private var stockCapitalGain: Double? = null
		private var stockDividendYield: Double? = null
		fun cashPart(cashPart: Double?) = apply { this.cashPart = cashPart }
		fun cashInterestRate(cashInterestRate: Double?) = apply { this.cashInterestRate = cashInterestRate }
		fun bondPart(bondPart: Double?) = apply { this.bondPart = bondPart }
		fun bondInterestRate(bondInterestRate: Double?) = apply { this.bondInterestRate = bondInterestRate }
		fun stockPart(stockPart: Double?) = apply { this.stockPart = stockPart }
		fun stockCapitalGain(stockCapitalGain: Double?) = apply { this.stockCapitalGain = stockCapitalGain }
		fun stockDividendYield(stockDividendYield: Double?) = apply { this.stockDividendYield = stockDividendYield }
		fun build(): AssetAllocation {
			return AssetAllocation(
				cashPart = cashPart ?: 0.0,
				cashInterestRate = cashInterestRate,
				bondPart = bondPart ?: 0.0,
				bondInterestRate = bondInterestRate,
				stockPart = stockPart ?: 0.0,
				stockCapitalGain = stockCapitalGain,
				stockDividendYield = stockDividendYield,
			)
		}
	}

}

/**
 * An investment is a portfolio of assets (maybe just one), with a cash part, a bond part and a stock part.
 * - cash part and the bond part generate interest
 * - the stock part generates dividends and capital gains
 * - investment can be managed by a fund manager, who may charge a management fee and a performance fee
 * - investment can be contributed to and withdrawn from
 * - investment can be owned by a person
 *
 * @property tag tag (non-unique) to map back to client entity
 * @property owner the owner of the investment account
 * @property reconBalance the balance of the investment account at the reconDate
 * @property contributionAmount the contribution amount at a given date
 * @property contributionPeriodicity the periodicity of the contribution
 * @property contributionStartDate the (optional) start date of the contribution, if omitted will start at month following reconDate
 * @property contributionEndDate the (optional) end date of the contribution, if omitted will continue indefinitely or bound by owner's deathDate (if any)
 * @property assetAllocation the asset allocation of the portfolio
 * @property interestUtilisation the use of the interest, either COMPOUNDed on the holding or DISBURSEd to the internalCash
 * @property dividendUtilisation the use of the dividend, either COMPOUNDed on the holding or DISBURSEd to the internalCash
 * @property withdrawalAmount the withdrawal amount at a given date, which is either a percentage of the portfolio (0 .. 1.0) or a fixed amount (if > 1.0)
 * @property withdrawalPeriodicity the periodicity of the withdrawal
 * @property withdrawalStartDate the (optional) start date of the withdrawal, if omitted will start at month following reconDate
 * @property withdrawalEndDate the (optional) end date of the withdrawal, if omitted will continue indefinitely or bound by owner's deathDate (if any)
 * @property managementFee the management fee of the portfolio (if any)
 * @property performanceFee the performance fee of the portfolio (if any)
 */
data class Investment(
	override val tag: String,
	override val owner: Person? = null,
	override val taxCode: String? = null,
	override val reconBalance: Double = 0.0,

	val contributionAmount: ValueProvider<Double>? = null,
	val contributionPeriodicity: Periodicity = Periodicity.YEARLY,
	val contributionStartDate: SimDate? = null,
	val contributionEndDate: SimDate? = null,

	val assetAllocation: AssetAllocation,
	val interestUtilisation: FundsUtilisation = FundsUtilisation.COMPOUND,
	val dividendUtilisation: FundsUtilisation = FundsUtilisation.COMPOUND,

	val withdrawalAmount: ValueProvider<Double>? = null,
	val withdrawalPeriodicity: Periodicity = Periodicity.YEARLY,
	val withdrawalStartDate: SimDate? = null,
	val withdrawalEndDate: SimDate? = null,

	val managementFee: Double? = 0.0,
	val performanceFee: Double? = 0.0,
) : ValuableBase(tag, owner, ProductType.ASSET, taxCode, reconBalance) {

	class Builder {
		private var tag: String? = null
		private var taxCode: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var contributionAmount: ValueProvider<Double>? = null
		private var contributionPeriodicity: Periodicity? = null
		private var contributionStartDate: SimDate? = null
		private var contributionEndDate: SimDate? = null
		private var assetAllocation: AssetAllocation? = null
		private var interestUtilisation: FundsUtilisation? = null
		private var dividendUtilisation: FundsUtilisation? = null
		private var withdrawalAmount: ValueProvider<Double>? = null
		private var withdrawalPeriodicity: Periodicity? = null
		private var withdrawalStartDate: SimDate? = null
		private var withdrawalEndDate: SimDate? = null
		private var managementFee: Double? = null
		private var performanceFee: Double? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun taxCode(taxCode: String?) = apply { this.taxCode = taxCode }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun contributionAmount(contributionAmount: ValueProvider<Double>?) = apply { this.contributionAmount = contributionAmount }
		fun contributionPeriodicity(contributionPeriodicity: Periodicity?) = apply { this.contributionPeriodicity = contributionPeriodicity }
		fun contributionStartDate(contributionStartDate: SimDate?) = apply { this.contributionStartDate = contributionStartDate }
		fun contributionEndDate(contributionEndDate: SimDate?) = apply { this.contributionEndDate = contributionEndDate }
		fun assetAllocation(assetAllocation: AssetAllocation?) = apply { this.assetAllocation = assetAllocation }
		fun interestUtilisation(interestUtilisation: FundsUtilisation?) = apply { this.interestUtilisation = interestUtilisation }
		fun dividendUtilisation(dividendUtilisation: FundsUtilisation?) = apply { this.dividendUtilisation = dividendUtilisation }
		fun withdrawalAmount(withdrawalAmount: ValueProvider<Double>?) = apply { this.withdrawalAmount = withdrawalAmount }
		fun withdrawalPeriodicity(withdrawalPeriodicity: Periodicity?) = apply { this.withdrawalPeriodicity = withdrawalPeriodicity }
		fun withdrawalStartDate(withdrawalStartDate: SimDate?) = apply { this.withdrawalStartDate = withdrawalStartDate }
		fun withdrawalEndDate(withdrawalEndDate: SimDate?) = apply { this.withdrawalEndDate = withdrawalEndDate }
		fun managementFee(managementFee: Double?) = apply { this.managementFee = managementFee }
		fun performanceFee(performanceFee: Double?) = apply { this.performanceFee = performanceFee }

		fun build(): Investment {
			return Investment(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				taxCode = taxCode,
				owner = owner,
				reconBalance = reconBalance ?: 0.0,
				//
				contributionAmount = contributionAmount,
				contributionPeriodicity = contributionPeriodicity ?: Periodicity.YEARLY,
				contributionStartDate = contributionStartDate,
				contributionEndDate = contributionEndDate,
				//
				assetAllocation = assetAllocation ?: throw IllegalArgumentException("assetAllocation is required"),
				interestUtilisation = interestUtilisation ?: FundsUtilisation.COMPOUND,
				dividendUtilisation = dividendUtilisation ?: FundsUtilisation.COMPOUND,
				//
				withdrawalAmount = withdrawalAmount,
				withdrawalPeriodicity = withdrawalPeriodicity ?: Periodicity.YEARLY,
				withdrawalStartDate = withdrawalStartDate,
				withdrawalEndDate = withdrawalEndDate,
				//
				managementFee = managementFee ?: 0.0,
				performanceFee = performanceFee ?: 0.0,
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		val compos: MutableList<FinancialCompo> = mutableListOf()
		if (contributionAmount != null) {
			compos.add(
				TransferCompo(
					holding = this,
					tag = "contribution",
					fromHolding = household.internalCash,
					toHolding = this,
					amount = contributionAmount,
					startDate = contributionStartDate,
					endDate = contributionEndDate,
					periodicity = contributionPeriodicity,
				)
			)
		}
		if (assetAllocation.cashPart != 0.0) {
			if (assetAllocation.cashInterestRate != null && assetAllocation.cashInterestRate != 0.0) {
				compos.add(
					InterestCompo(
						holding = this,
						tag = "cashInterest",
						interestRate = constValue(assetAllocation.cashPart / 100.0 * assetAllocation.cashInterestRate),
						interestPeriodicity = Periodicity.YEARLY,
						fundsUtilisation = interestUtilisation,
					)
				)
			}
		}
		if (assetAllocation.bondPart != 0.0) {
			if (assetAllocation.bondInterestRate != null && assetAllocation.bondInterestRate != 0.0) {
				compos.add(
					InterestCompo(
						holding = this,
						tag = "bondInterest",
						interestRate = constValue(assetAllocation.bondPart / 100.0 * assetAllocation.bondInterestRate),
						interestPeriodicity = Periodicity.YEARLY,
						fundsUtilisation = interestUtilisation,
					)
				)
			}
		}
		if (assetAllocation.stockPart != 0.0) {
			if (assetAllocation.stockDividendYield != null && assetAllocation.stockDividendYield != 0.0) {
				compos.add(
					DividendCompo(
						holding = this,
						dividendYield = constValue(assetAllocation.stockPart / 100.0 * assetAllocation.stockDividendYield),
						dividendPeriodicity = Periodicity.YEARLY,
						fundsUtilisation = dividendUtilisation,
					)
				)
			}
			if (assetAllocation.stockCapitalGain != null && assetAllocation.stockCapitalGain != 0.0) {
				compos.add(
					CapitalGainCompo(
						holding = this,
						capitalGainRate = constValue(assetAllocation.stockPart / 100.0 * assetAllocation.stockCapitalGain),
					)
				)
			}
		}
		if (performanceFee != null && performanceFee != 0.0) {
			compos.add(
				CashflowCompo(
					holding = this,
					tag = "performanceFee",
					fundsUtilisation = FundsUtilisation.COMPOUND,
					direction = FlowDirection.OUTGOING,
					amount = ValueProvider { _, date ->
						val sop = SimDate.max(date - 12, household.reconDate)
						val delta = this.getBalance(date) - this.getBalance(sop)
						return@ValueProvider performanceFee / 100 * delta
					},
					periodicity = Periodicity.YEARLY,
					startDate = household.reconDate + 1,
				)
			)
		}
		if (managementFee != null && managementFee != 0.0) {
			compos.add(
				CashflowCompo(
					holding = this,
					tag = "managementFee",
					fundsUtilisation = FundsUtilisation.COMPOUND,
					direction = FlowDirection.OUTGOING,
					amount = ValueProvider { compo, date ->
						return@ValueProvider (managementFee / 100.0 * compo.holding.getBalance(date - 1))
					},
					periodicity = Periodicity.YEARLY,
					startDate = household.reconDate + 1,
				)
			)
		}
		if (withdrawalAmount != null) {
			compos.add(
				TransferCompo(
					holding = this,
					tag = "withdrawal",
					fromHolding = this,
					toHolding = household.internalCash,
					amount = withdrawalAmount,
					startDate = withdrawalStartDate,
					endDate = withdrawalEndDate,
					periodicity = withdrawalPeriodicity,
				)
			)
		}
		return compos
	}

}
