package io.fincast.holding.impl.core

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProvider
import io.fincast.compo.impl.CashflowCompo
import io.fincast.compo.impl.FlowDirection
import io.fincast.engine.SimDate
import io.fincast.enums.FundsUtilisation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.holding.Contract
import io.fincast.holding.base.HoldingBase
import io.fincast.household.Person

/**
 * Personal or Household Expense
 *
 * Generates a periodic outgoing cashflow (from internalCash to externalCash) between startDate and endDate.
 *
 * The expense can be attached to a person, in which case the expense is bound by the person's death date.
 *
 * @property tag tag (non-unique) to map back to client entity
 * @property owner the (optional) owner of the holding
 * @property amount the amount of expense at a given date
 * @property periodicity the periodicity of the expense
 * @property startDate the (optional) start date of the expense, if omitted will start at month following reconDate
 * @property endDate the (optional) end date of the expense
 * - when no endDate is given, it will be the owners deathDate (if any)
 * - when an endDate is given, it will be the least of the given endDate and the owners deathDate (if any)
 */
data class Expense(
	override val tag: String,
	override val owner: Person? = null,
	override val taxCode: String? = null,
	val amount: ValueProvider<Double>,
	val periodicity: Periodicity = Periodicity.YEARLY,
	override val startDate: SimDate? = null,
	override val endDate: SimDate? = null,
) : Contract, HoldingBase(tag, owner, ProductType.EXPENSE, taxCode) {

	class Builder {
		private var tag: String? = null
		private var taxCode: String? = null
		private var owner: Person? = null
		private var amount: ValueProvider<Double>? = null
		private var periodicity: Periodicity? = null
		private var startDate: SimDate? = null
		private var endDate: SimDate? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun taxCode(taxCode: String?) = apply { this.taxCode = taxCode }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun amount(amount: ValueProvider<Double>) = apply { this.amount = amount }
		fun periodicity(periodicity: Periodicity?) = apply { this.periodicity = periodicity }
		fun startDate(startDate: SimDate?) = apply { this.startDate = startDate }
		fun endDate(endDate: SimDate?) = apply { this.endDate = endDate }
		fun build(): Expense {
			return Expense(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				taxCode = taxCode,
				owner = owner,
				amount = amount ?: throw IllegalArgumentException("amount is required"),
				periodicity = periodicity ?: Periodicity.YEARLY,
				startDate = startDate,
				endDate = endDate,
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		val endDate = owner?.getEndOfExpenseDate(endDate) ?: endDate
		val cf = CashflowCompo(
			holding = this,
			tag = "expense",
			fundsUtilisation = FundsUtilisation.DISBURSE,
			direction = FlowDirection.OUTGOING,
			amount = amount,
			startDate = startDate,
			endDate = endDate,
			periodicity = periodicity,
		)
		return listOf(cf)
	}

}
