package io.fincast.holding.impl.core

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProviders.constValue
import io.fincast.compo.ValueProviders.prevEomCreditBalance
import io.fincast.compo.ValueProviders.prevEomDebitBalance
import io.fincast.compo.impl.InterestCompo
import io.fincast.enums.FundsUtilisation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.holding.base.ValuableBase
import io.fincast.household.Person

/**
 * A generic bank account.
 * Unlike other holdings, bank accounts can have both a negative and a positive balance, i.e. be an asset or liability.
 * As a consequence, they may have both credit and debit interest rates.
 * Interest payments are always compounded.
 *
 * @property tag tag (non-unique) to map back to client entity
 * @property owner the owner of the bank account
 * @property reconBalance the balance of the bank account at the reconDate
 * @property creditInterestRate the credit interest rate of the bank account (if any)
 * @property debitInterestRate the debit interest rate of the bank account (if any)
 * @property interestPeriodicity the periodicity of the interest payments
 */
data class BankAccount(
	override val tag: String,
	override val owner: Person? = null,
	override val taxCode: String? = null,
	override val reconBalance: Double = 0.0,
	val creditInterestRate: Double? = 0.0,
	val debitInterestRate: Double? = 0.0,
	val interestPeriodicity: Periodicity = Periodicity.YEARLY,
) : ValuableBase(tag, owner, ProductType.VALUABLE, taxCode, reconBalance) {

	class Builder {
		private var tag: String? = null
		private var taxCode: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var creditInterestRate: Double? = null
		private var debitInterestRate: Double? = null
		private var interestPeriodicity: Periodicity? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun taxCode(taxCode: String?) = apply { this.taxCode = taxCode }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun creditInterestRate(creditInterestRate: Double?) = apply { this.creditInterestRate = creditInterestRate }
		fun debitInterestRate(debitInterestRate: Double?) = apply { this.debitInterestRate = debitInterestRate }
		fun interestPeriodicity(interestPeriodicity: Periodicity?) = apply { this.interestPeriodicity = interestPeriodicity }
		fun build(): BankAccount {
			return BankAccount(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				taxCode = taxCode,
				owner = owner,
				reconBalance = reconBalance ?: 0.0,
				creditInterestRate = creditInterestRate ?: 0.0,
				debitInterestRate = debitInterestRate ?: 0.0,
				interestPeriodicity = interestPeriodicity ?: Periodicity.YEARLY,
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		val compos = mutableListOf<FinancialCompo>()
		if (creditInterestRate != null && creditInterestRate != 0.0) {
			compos.add(
				InterestCompo(
					holding = this,
					tag = "creditInterest",
					amount = prevEomCreditBalance(),
					interestRate = constValue(creditInterestRate),
					interestPeriodicity = interestPeriodicity,
					fundsUtilisation = FundsUtilisation.COMPOUND,
				)
			)
		}
		if (debitInterestRate != null && debitInterestRate != 0.0) {
			compos.add(
				InterestCompo(
					holding = this,
					tag = "debitInterest",
					amount = prevEomDebitBalance(),
					interestRate = constValue(debitInterestRate),
					interestPeriodicity = interestPeriodicity,
					fundsUtilisation = FundsUtilisation.COMPOUND,
				)
			)
		}
		return compos
	}

}
