package io.fincast.holding.impl.ch

import io.fincast.compo.FinancialCompo
import io.fincast.engine.Booking
import io.fincast.engine.ProjectionListener
import io.fincast.engine.SimDate
import io.fincast.enums.BookingKind
import io.fincast.enums.ProductType
import io.fincast.holding.Contract
import io.fincast.holding.Holding
import io.fincast.holding.TaxHandler
import io.fincast.holding.base.HoldingBase
import io.fincast.holding.impl.core.Expense
import io.fincast.holding.impl.core.Income
import io.fincast.holding.impl.core.Investment
import io.fincast.holding.impl.core.RealEstate
import io.fincast.spi.ChTaxHouseholdInfo
import io.fincast.spi.ChTaxPersonalInfo
import io.fincast.spi.ChTaxService
import kotlin.math.round

data class HouseholdExpenses(
	val insurance: Double = 0.0,
	val childCareCosts: Double = 0.0,
	val other: Double = 0.0,
)

data class PersonalExpenses(
	val primaryOccupationMealCosts: Double = 0.0,
	val primaryOccupationTravelCosts: Double = 0.0,
	val primaryOccupationOtherExpenses: Double = 0.0,
	val secondaryOccupationExpenses: Double = 0.0,
)

data class IncomeTaxAggregation(
	var p1PrimarySalary: Double = 0.0,
	var p1SecondarySalary: Double = 0.0,
	var p1PensionIncome: Double = 0.0,
	var p1PensionPayment: Double = 0.0,
	var p2PrimarySalary: Double = 0.0,
	var p2SecondarySalary: Double = 0.0,
	var p2PensionIncome: Double = 0.0,
	var p2PensionPayment: Double = 0.0,
	var hhInvestmentIncome: Double = 0.0,
	var hhInvestmentIncomeParticipations: Double = 0.0,
	var hhRentalExpense: Double = 0.0,
	var hhRentalIncome: Double = 0.0,
	var hhDebtInterest: Double = 0.0,
	var hhBuildingMaintenanceCosts: Double = 0.0,
	var hhPillar3aContributions: Double = 0.0,
)

data class ChTax(
	override val tag: String,
	val householdExpenses: HouseholdExpenses = HouseholdExpenses(),
	val partner1Expenses: PersonalExpenses = PersonalExpenses(),
	val partner2Expenses: PersonalExpenses? = null,
) : Contract, HoldingBase(tag, null, ProductType.EXPENSE, null), ProjectionListener, TaxHandler {

	override val startDate: SimDate? = null
	override val endDate: SimDate? = null

	private var isActive = false
	var taxAggregation = IncomeTaxAggregation()

	class Builder {
		private var tag: String? = null
		private var householdExpenses: HouseholdExpenses? = null
		private var partner1Expenses: PersonalExpenses? = null
		private var partner2Expenses: PersonalExpenses? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun householdExpenses(householdExpenses: HouseholdExpenses?) = apply { this.householdExpenses = householdExpenses }
		fun partner1Expenses(partner1Deductions: PersonalExpenses?) = apply { this.partner1Expenses = partner1Deductions }
		fun partner2Expenses(partner2Deductions: PersonalExpenses?) = apply { this.partner2Expenses = partner2Deductions }
		fun build(): ChTax {
			return ChTax(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				householdExpenses = householdExpenses ?: throw IllegalArgumentException("householdDeductions is required"),
				partner1Expenses = partner1Expenses ?: throw IllegalArgumentException("partner1Deductions is required"),
				partner2Expenses = partner2Expenses
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		return emptyList()
	}

	// will not calculate tax in first projection year
	// TODO: handle first year tax calculation
	override fun onStartOfMonth(date: SimDate) {
		isActive = isActive || date.isStartOfYear
		if (date.isStartOfYear) {
			taxAggregation = IncomeTaxAggregation()
		}
	}

	// Income Tax Aggregation
	override fun onBooking(booking: Booking) {

		if (!isActive || booking !is Booking.Lifecycle) {
			return
		}

		val bookingKind = booking.bookingKind
		val amount = booking.amount
		val holding = booking.holding
		val trigHolding = booking.trigHolding

		// INCOME (FROM EXTERNAL CASH)
		if (holding == household.externalCash && amount < 0) {
			if (trigHolding is Income) {
				if (trigHolding.owner == household.partner1) {
					if (trigHolding.taxCode == SalaryPrimary) {
						taxAggregation.p1PrimarySalary -= amount
					} else {
						taxAggregation.p1SecondarySalary -= amount
					}
				} else {
					if (trigHolding.taxCode == SalaryPrimary) {
						taxAggregation.p2PrimarySalary -= amount
					} else {
						taxAggregation.p2SecondarySalary -= amount
					}
				}
			} else if (trigHolding is ChPillarOne || trigHolding is ChPillarTwoPension) {
				if (trigHolding.owner == household.partner1) {
					taxAggregation.p1PensionIncome -= amount
				} else {
					taxAggregation.p2PensionIncome -= amount
				}
			} else if (trigHolding is RealEstate) {
				taxAggregation.hhRentalIncome -= amount
			} else if (bookingKind == BookingKind.INTEREST || bookingKind == BookingKind.DIVIDEND) {
				taxAggregation.hhInvestmentIncome -= amount
			}
		}

		// DEDUCTABLE EXPENSES
		if (holding == household.externalCash && amount > 0) {
			if (trigHolding is RealEstate) {
				taxAggregation.hhBuildingMaintenanceCosts += amount
			} else if (trigHolding is Expense && trigHolding.taxCode == ExpenseRent) {
				taxAggregation.hhRentalExpense += amount
			} else if (bookingKind == BookingKind.INTEREST) {
				taxAggregation.hhDebtInterest += amount
			}
		}

		// CONTRIBUTIONS TO / PAYMENTS FROM PILLAR 2
		if ((holding is ChPillarTwoCapital || holding is ChPillarTwoPension) && bookingKind == BookingKind.TRANSFER) {
			if (amount > 0) {
				taxAggregation.hhPillar3aContributions += amount
			} else {
				if (trigHolding.owner == household.partner1) {
					taxAggregation.p1PensionPayment -= amount
				} else {
					taxAggregation.p2PensionPayment -= amount
				}
			}
		}

		// CONTRIBUTIONS TO / PAYMENTS FROM PILLAR 3A
		if (holding is Investment && holding.taxCode == Investment3a && bookingKind == BookingKind.TRANSFER) {
			if (amount > 0) {
				taxAggregation.hhPillar3aContributions += amount
			} else {
				if (trigHolding.owner == household.partner1) {
					taxAggregation.p1PensionPayment -= amount
				} else {
					taxAggregation.p2PensionPayment -= amount
				}
			}
		}

	}

	override fun handleTaxes(date: SimDate, taxService: ChTaxService) {

		if (!isActive || !date.isEndOfYear) {
			return
		}

		taxAggregation.p1PrimarySalary = round(taxAggregation.p1PrimarySalary)
		taxAggregation.p1SecondarySalary = round(taxAggregation.p1SecondarySalary)
		taxAggregation.p1PensionIncome = round(taxAggregation.p1PensionIncome)
		taxAggregation.p1PensionPayment = round(taxAggregation.p1PensionPayment)
		taxAggregation.p2PrimarySalary = round(taxAggregation.p2PrimarySalary)
		taxAggregation.p2SecondarySalary = round(taxAggregation.p2SecondarySalary)
		taxAggregation.p2PensionIncome = round(taxAggregation.p2PensionIncome)
		taxAggregation.p2PensionPayment = round(taxAggregation.p2PensionPayment)
		taxAggregation.hhInvestmentIncome = round(taxAggregation.hhInvestmentIncome)
		taxAggregation.hhInvestmentIncomeParticipations = round(taxAggregation.hhInvestmentIncomeParticipations)
		taxAggregation.hhRentalExpense = round(taxAggregation.hhRentalExpense)
		taxAggregation.hhRentalIncome = round(taxAggregation.hhRentalIncome)
		taxAggregation.hhDebtInterest = round(taxAggregation.hhDebtInterest)
		taxAggregation.hhBuildingMaintenanceCosts = round(taxAggregation.hhBuildingMaintenanceCosts)
		taxAggregation.hhPillar3aContributions = round(taxAggregation.hhPillar3aContributions)

		val partner1 = household.partner1
		val partner2 = household.partner2

		val householdInfo = ChTaxHouseholdInfo(
			otherIncome = 0.0,
			netWealth = this.taxableNetWealth(),
			insurance = this.householdExpenses.insurance,
			childCareCosts = this.householdExpenses.childCareCosts,
			otherDeductions = this.householdExpenses.other,
			rent = taxAggregation.hhRentalExpense,
			investmentIncome = taxAggregation.hhInvestmentIncome,
			rentalIncome = taxAggregation.hhRentalIncome,
			debtInterestExpenses = taxAggregation.hhDebtInterest,
			realEstateMaintenance = taxAggregation.hhBuildingMaintenanceCosts,
			pillar3aContributions = taxAggregation.hhPillar3aContributions,
		)

		val p1HasPrimaryIncome = taxAggregation.p1PrimarySalary > 0.0
		val p1HasSecondaryIncome = taxAggregation.p1SecondarySalary > 0.0
		val partner1Info = ChTaxPersonalInfo(
			age = partner1.getAgeAt(date),
			confession = household.partner1.religiousAffiliation,
			primaryOccupationMealCosts = if (p1HasPrimaryIncome) this.partner1Expenses.primaryOccupationMealCosts else 0.0,
			primaryOccupationTravelCosts = if (p1HasPrimaryIncome) this.partner1Expenses.primaryOccupationTravelCosts else 0.0,
			primaryOccupationOtherExpenses = if (p1HasPrimaryIncome) this.partner1Expenses.primaryOccupationOtherExpenses else 0.0,
			secondaryOccupationExpenses = if (p1HasSecondaryIncome) this.partner1Expenses.secondaryOccupationExpenses else 0.0,
			primaryNetIncome = taxAggregation.p1PrimarySalary + taxAggregation.p1PensionIncome,
			secondaryNetIncome = taxAggregation.p1SecondarySalary,
			pensionPayment = taxAggregation.p1PensionPayment,
		)

		val p2HasPrimaryIncome = taxAggregation.p2PrimarySalary > 0.0
		val p2HasSecondaryIncome = taxAggregation.p2SecondarySalary > 0.0
		val partner2Info = if (partner2 == null) null else ChTaxPersonalInfo(
			age = partner2.getAgeAt(date),
			confession = partner2.religiousAffiliation,
			primaryOccupationMealCosts = if (p2HasPrimaryIncome) this.partner2Expenses?.primaryOccupationMealCosts
				?: 0.0 else 0.0,
			primaryOccupationTravelCosts = if (p2HasPrimaryIncome) this.partner2Expenses?.primaryOccupationTravelCosts
				?: 0.0 else 0.0,
			primaryOccupationOtherExpenses = if (p2HasPrimaryIncome) this.partner2Expenses?.primaryOccupationOtherExpenses
				?: 0.0 else 0.0,
			secondaryOccupationExpenses = if (p2HasSecondaryIncome) this.partner2Expenses?.secondaryOccupationExpenses
				?: 0.0 else 0.0,
			primaryNetIncome = taxAggregation.p2PrimarySalary + taxAggregation.p2PensionIncome,
			secondaryNetIncome = taxAggregation.p2SecondarySalary,
			pensionPayment = taxAggregation.p2PensionPayment,
		)

		val taxResult = taxService.getTaxes(
			taxYear = date.year,
			taxLocationId = household.zip * 100000,
			civilStatus = household.civilStatus,
			householdInfo = householdInfo,
			partner1Info = partner1Info,
			partner2Info = partner2Info,
			children = emptyList()
		)

		if (taxResult.incomeTax > 0.0) {
			household.internalCash.bookCashflow(
				date = date,
				bookingKind = BookingKind.CASHFLOW,
				amount = -taxResult.incomeTax,
				trigHolding = this,
				trigCompo = "incomeTax",
			)
		}
		if (taxResult.wealthTax > 0.0) {
			household.internalCash.bookCashflow(
				date = date,
				bookingKind = BookingKind.CASHFLOW,
				amount = -taxResult.wealthTax,
				trigHolding = this,
				trigCompo = "wealthTax",
			)
		}
		if (taxResult.capitalTax > 0.0) {
			household.internalCash.bookCashflow(
				date = date,
				bookingKind = BookingKind.CASHFLOW,
				amount = -taxResult.capitalTax,
				trigHolding = this,
				trigCompo = "capitalTax",
			)
		}

	}

	private fun taxableNetWealth(): Double {
		return (household.holdings + household.internalCash)
			.filter { isTaxableHolding(it) }
			.map { it.getBalance() }
			.sum()
	}

	private fun isTaxableHolding(holding: Holding): Boolean {
		if (!holding.productType.isValuable) {
			return false
		} else if (holding is ChPillarTwoCapital) {
			return false
		} else if (holding is Investment && holding.taxCode == Investment3a) {
			return false
		}
		return true
	}

	companion object {
		val SalaryPrimary = "primarySalary"
		val SalarySecondary = "secondarySalary"
		val ExpenseRent = "rent"
		val ExpenseAlimonyMinor = "alimonyMinor"
		val ExpenseAlimonyAdult = "alimonyAdult"
		val Investment3a = "3a"
	}

}
