package io.fincast.holding.impl.ch

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProviders.constValue
import io.fincast.compo.impl.CashflowCompo
import io.fincast.compo.impl.FlowDirection
import io.fincast.engine.SimDate
import io.fincast.enums.FundsUtilisation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.holding.Contract
import io.fincast.holding.base.HoldingBase
import io.fincast.household.Person
import kotlin.math.min

/**
 * Swiss 2nd pillar pension plan (BVG), pension part.
 *
 * @property tag tag (non-unique) to map back to client entity
 * @property owner the owner of the pension plan
 * @property vestedBenefits the accumulated vested benefits at the reconDate
 * @property projectedVestedBenefits the projected vested benefits at the retirementDate
 * @property capitalWithdrawal the planned capital withdrawal at retirementDate, in % of vested benefits (0 .. 1.0) or in CHF
 * @property conversionRate the conversion rate from vested benefits to pension
 */
data class ChPillarTwoPension(
	override val tag: String,
	override val owner: Person,
	override val taxCode: String? = null,
	val vestedBenefits: Double = 0.0,
	val projectedVestedBenefits: Double = 0.0,
	val capitalWithdrawal: Double = 0.0,
	val conversionRate: Double = 0.0,
) : Contract, HoldingBase(tag, owner, ProductType.CONTRACT, taxCode) {

	override val startDate: SimDate? = null
	override val endDate: SimDate? = null

	class Builder {
		private var tag: String? = null
		private var taxCode: String? = null
		private var owner: Person? = null
		private var vestedBenefits: Double? = null
		private var projectedVestedBenefits: Double? = null
		private var capitalWithdrawal: Double? = null
		private var conversionRate: Double? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun taxCode(taxCode: String?) = apply { this.taxCode = taxCode }
		fun owner(owner: Person) = apply { this.owner = owner }
		fun vestedBenefits(vestedBenefits: Double?) = apply { this.vestedBenefits = vestedBenefits }
		fun projectedVestedBenefits(vestedBenefits: Double?) = apply { this.projectedVestedBenefits = vestedBenefits }
		fun capitalWithdrawal(capitalWithdrawal: Double?) = apply { this.capitalWithdrawal = capitalWithdrawal }
		fun conversionRate(conversionRate: Double?) = apply { this.conversionRate = conversionRate }

		fun build(): ChPillarTwoPension {
			return ChPillarTwoPension(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				taxCode = taxCode,
				owner = owner ?: throw IllegalArgumentException("owner is required"),
				vestedBenefits = vestedBenefits ?: 0.0,
				projectedVestedBenefits = projectedVestedBenefits ?: 0.0,
				capitalWithdrawal = capitalWithdrawal ?: 0.0,
				conversionRate = conversionRate ?: 0.0,
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		val retirementDate = owner.getActualRetirementDate()
		val deathDate = owner.deathDate
		if (deathDate != null && deathDate <= retirementDate) {
			return emptyList()
		} else if (projectedVestedBenefits == 0.0) {
			return emptyList()
		} else if (capitalWithdrawal == 1.0 || capitalWithdrawal >= projectedVestedBenefits) {
			return emptyList()
		}
		val capitalWithdrawal = if (capitalWithdrawal <= 1.0) (capitalWithdrawal * projectedVestedBenefits) else min(capitalWithdrawal, projectedVestedBenefits)
		val pensionBase = projectedVestedBenefits - capitalWithdrawal
		if (pensionBase > 0.0) {
			val compo = CashflowCompo(
				holding = this,
				tag = "pension",
				fundsUtilisation = FundsUtilisation.DISBURSE,
				direction = FlowDirection.INCOMING,
				amount = constValue(pensionBase * conversionRate / 100.0 / 12),
				startDate = retirementDate + 1,
				endDate = deathDate,
				periodicity = Periodicity.MONTHLY,
			)
			return listOf(compo)
		}
		return emptyList()
	}

}
