package io.fincast.holding.impl.ch

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProviders.constValue
import io.fincast.compo.impl.CashflowCompo
import io.fincast.compo.impl.FlowDirection
import io.fincast.compo.impl.TransferCompo
import io.fincast.enums.FundsUtilisation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.holding.base.ValuableBase
import io.fincast.household.Person
import kotlin.math.min

/**
 * Swiss 2nd pillar pension plan (BVG), capital part.
 *
 * @property tag tag (non-unique) to map back to client entity
 * @property owner the owner of the pension plan
 * @property vestedBenefits the accumulated vested benefits at the reconDate
 * @property projectedVestedBenefits the projected vested benefits at the retirementDate
 * @property capitalWithdrawal the planned capital withdrawal at retirementDate, in % of vested benefits (0 .. 1.0) or in CHF
 */
data class ChPillarTwoCapital(
	override val tag: String,
	override val owner: Person,
	override val taxCode: String? = null,
	val vestedBenefits: Double = 0.0,
	val projectedVestedBenefits: Double = 0.0,
	val capitalWithdrawal: Double = 0.0,
) : ValuableBase(tag, owner, ProductType.ASSET, taxCode, vestedBenefits) {

	override val reconBalance get() = capitalVestedBenefits

	class Builder {
		private var tag: String? = null
		private var taxCode: String? = null
		private var owner: Person? = null
		private var vestedBenefits: Double? = null
		private var projectedVestedBenefits: Double? = null
		private var capitalWithdrawal: Double? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun taxCode(taxCode: String?) = apply { this.taxCode = taxCode }
		fun owner(owner: Person) = apply { this.owner = owner }
		fun vestedBenefits(vestedBenefits: Double?) = apply { this.vestedBenefits = vestedBenefits }
		fun projectedVestedBenefits(vestedBenefits: Double?) = apply { this.projectedVestedBenefits = vestedBenefits }
		fun capitalWithdrawal(capitalWithdrawal: Double?) = apply { this.capitalWithdrawal = capitalWithdrawal }

		fun build(): ChPillarTwoCapital {
			return ChPillarTwoCapital(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				taxCode = taxCode,
				owner = owner ?: throw IllegalArgumentException("owner is required"),
				vestedBenefits = vestedBenefits ?: 0.0,
				projectedVestedBenefits = projectedVestedBenefits ?: 0.0,
				capitalWithdrawal = capitalWithdrawal ?: 0.0,
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		val retirementDate = owner.getActualRetirementDate()
		val deathDate = owner.deathDate
		if (deathDate != null && deathDate <= retirementDate) {
			return emptyList()
		} else if (projectedVestedBenefits == 0.0) {
			return emptyList()
		} else if (capitalWithdrawal == 0.0) {
			return emptyList()
		}
		val capitalWithdrawal = if (capitalWithdrawal <= 1.0) (capitalWithdrawal * projectedVestedBenefits) else min(capitalWithdrawal, projectedVestedBenefits)
		val compos: MutableList<FinancialCompo> = mutableListOf()
		val endDate = owner.getEndOfExpenseDate(retirementDate) ?: retirementDate
		val monthlyContribution = (capitalWithdrawal - capitalVestedBenefits) / (retirementDate - household.reconDate)
		compos.add(
			CashflowCompo(
				holding = this,
				tag = "capitalContribution",
				fundsUtilisation = FundsUtilisation.COMPOUND,
				direction = FlowDirection.INCOMING,
				amount = constValue(monthlyContribution),
				endDate = endDate,
				periodicity = Periodicity.MONTHLY,
			)
		)
		compos.add(
			TransferCompo(
				holding = this,
				tag = "capitalWithdrawal",
				fromHolding = this,
				toHolding = household.internalCash,
				amount = constValue(capitalWithdrawal),
				endDate = retirementDate + 1,
				periodicity = Periodicity.ONCE,
			)
		)
		return compos
	}

	private val capitalVestedBenefits: Double
		get() {
			val capitalQuota = if (capitalWithdrawal <= 1) capitalWithdrawal else capitalWithdrawal / projectedVestedBenefits
			return capitalQuota * vestedBenefits
		}

}
