package io.fincast.holding.base

import io.fincast.compo.FinancialCompo
import io.fincast.engine.Booking
import io.fincast.engine.BookingListener
import io.fincast.engine.SimDate
import io.fincast.enums.BookingKind
import io.fincast.enums.BookingType
import io.fincast.enums.ProductType
import io.fincast.holding.Holding
import io.fincast.holding.ProjectionPhase
import io.fincast.household.Household
import io.fincast.household.Person

abstract class HoldingBase(
	override val tag: String,
	override val owner: Person? = null,
	override val productType: ProductType,
	override val taxCode: String? = null,
) : Holding {

	private var _household: Household? = null

	override val household: Household by lazy { _household ?: throw IllegalStateException("household not set") }

	fun setHousehold(household: Household) {
		_household = household
	}

	override var compos: List<FinancialCompo> = emptyList()

	private var balance = 0.0

	private var reconDate: SimDate? = null

	private val balances: MutableMap<SimDate, Double> = mutableMapOf()

	private var bookingListener: BookingListener? = null

	override fun initProjection(bookingListener: BookingListener) {
		compos = createCompos()
		this.bookingListener = bookingListener
	}

	protected abstract fun createCompos(): List<FinancialCompo>

	override fun handleReconciliation(date: SimDate) {
	}

	override fun handleLifecycle(date: SimDate, projectionPhase: ProjectionPhase) {
		compos.filter { it.projectionPhase == projectionPhase }.forEach {
			//println("Handling ${projectionPhase} for ${this.tag} / ${this.productType} @ ${date}")
			it.handleLifecycle(date)
		}
	}

	override fun getBalance(): Double {
		return balance
	}

	override fun getBalance(date: SimDate): Double {
		return if (date < household.reconDate) {
			0.0
		} else if (!balances.containsKey(date)) {
			getBalance()
		} else {
			balances[date]!!
		}
	}

	override fun bookReconciliation(date: SimDate, balance: Double) {
		addBooking(Booking.Reconciliation(this, date, balance))
	}

	override fun bookTransfer(date: SimDate, amount: Double, counterHolding: Holding, trigHolding: Holding, trigCompo: String) {
		require(!this.isExternalCash && !counterHolding.isExternalCash) { "cannot transfer to/from external cash" }
		if (amount == 0.0) {
			return
		}
		this.addBooking(date, BookingKind.TRANSFER, amount, counterHolding, trigHolding, trigCompo)
		(counterHolding as HoldingBase).addBooking(date, BookingKind.TRANSFER, -amount, this, trigHolding, trigCompo)
	}

	override fun bookCashflow(date: SimDate, bookingKind: BookingKind, amount: Double, trigHolding: Holding, trigCompo: String) {
		require(this.productType.isValuable) { "can only book cashflow from/to valuable" }
		if (amount == 0.0) {
			return
		}
		this.addBooking(date, bookingKind, amount, household.externalCash, trigHolding, trigCompo)
		(household.externalCash as HoldingBase).addBooking(date, bookingKind, -amount, this, trigHolding, trigCompo)
	}

	private fun addBooking(date: SimDate, bookingKind: BookingKind, amount: Double, counterHolding: Holding, trigHolding: Holding, trigCompo: String) {
		addBooking(Booking.Lifecycle(this, date, bookingKind, amount, counterHolding, trigHolding, trigCompo))
	}

	private fun addBooking(booking: Booking) {
		val lastDate = getLastDate()
		require(lastDate == null || booking.date >= lastDate) { "bookings must be sequential i: ${booking.date} >= $lastDate" }
		require(lastDate == null || booking.date <= lastDate + 1) { "bookings must be sequential ii: ${booking.date} >= $lastDate" }
		require(reconDate == null || booking.date > reconDate!!) { "cannot add booking to reconciled period" }
		if (0.0 == booking.amount) {
			return
		}
		when (booking.bookingKind.bookingType) {
			BookingType.RECONCILE -> {
				balance = booking.amount
				reconDate = booking.date
			}

			BookingType.CASHFLOW, BookingType.TRANSFER -> {
				balance += booking.amount
			}
		}
		bookingListener!!.onBooking(booking)
	}

	override fun handleEndOfMonth(date: SimDate) {
		this.balances[date] = balance
	}

	private fun getLastDate(): SimDate? {
		return balances.keys.maxOrNull()
	}

}
