package io.fincast.holding

import io.fincast.compo.FinancialCompo
import io.fincast.engine.BookingListener
import io.fincast.engine.SimDate
import io.fincast.enums.BookingKind
import io.fincast.enums.ProductType
import io.fincast.household.Household
import io.fincast.household.Person

/**
 * Holding projection lifecycle phases:
 * - LIFECYCLE: Component Lifecycle (Cashflows, Interest, Dividends, ...) (during period)
 * - EOP_TRANSFER: Holding Transfers (unconditional, f.ex. percentage-based investments/redemptions) (end-of-period)
 * - EOY_TAX: Taxes (end-of-year)
 * - EOY_REBALANCE: Rebalancing Transfers (end-of-year)
 */
enum class ProjectionPhase {
	LIFECYCLE,
	EOP_TRANSFER,
	EOY_TAX,
	EOY_REBALANCE,
}

interface Holding {

	val household: Household

	val tag: String

	val owner: Person?

	val productType: ProductType

	val taxCode: String?

	val isAuxiliary: Boolean get() = isExternalCash || isInternalCash || isBufferInvestment

	val isExternalCash: Boolean get() = this == household.externalCash

	val isInternalCash: Boolean get() = this == household.internalCash

	val isBufferInvestment: Boolean get() = this == household.bufferInvestment

	fun initProjection(bookingListener: BookingListener)

	val compos: List<FinancialCompo>

	fun handleReconciliation(date: SimDate)

	fun handleLifecycle(date: SimDate, projectionPhase: ProjectionPhase)

	fun handleEndOfMonth(date: SimDate)

	fun getBalance(): Double

	fun getBalance(date: SimDate): Double

	fun getYield(): Double = compos.sumOf { it.getYield() }

	fun bookReconciliation(date: SimDate, balance: Double)

	fun bookTransfer(date: SimDate, amount: Double, counterHolding: Holding, trigHolding: Holding, trigCompo: String)

	fun bookCashflow(date: SimDate, bookingKind: BookingKind, amount: Double, trigHolding: Holding, trigCompo: String)

}
