package io.fincast.engine.impl

import io.fincast.compo.ValueProviders
import io.fincast.engine.*
import io.fincast.enums.BookingKind
import io.fincast.enums.BookingType
import io.fincast.enums.CivilStatus
import io.fincast.enums.Periodicity
import io.fincast.holding.Holding
import io.fincast.holding.ProjectionPhase
import io.fincast.holding.TaxHandler
import io.fincast.holding.Valuable
import io.fincast.household.Household
import io.fincast.spi.*
import kotlin.math.max
import kotlin.math.min

class ProjectionRunImpl(
	override val engine: ProjectionEngine,
	override val household: Household,
	override val holding: Holding?,
	override val periodicity: Periodicity,
	override val endDate: SimDate,
) : ProjectionRun, ProjectionListener, ChTaxService {

	private val holdings: List<Holding>
	private val projectionListeners: MutableList<ProjectionListener> = mutableListOf()

	private val projectionPeriods: MutableList<ProjectionPeriod> = mutableListOf()

	init {

		check(periodicity == Periodicity.MONTHLY || periodicity == Periodicity.YEARLY) { "periodicity must be MONTHLY or YEARLY" }

		val auxHoldings = listOf(household.internalCash, household.externalCash, household.bufferInvestment)
		holdings = (if (holding == null) household.holdings else listOf(holding)) + auxHoldings
		holdings.forEach { it.initProjection(this) }

		// register all listeners
		addProjectionListener(this)
		holdings.filterIsInstance<ProjectionListener>().forEach { listener -> addProjectionListener(listener) }

	}

	private var projectionPeriod: ProjectionPeriod? = null

	override fun calcProjection() {

		// reconciliation (first period)
		val reconDate = household.reconDate
		projectionListeners.forEach { it.onStartOfMonth(reconDate) }
		holdings.forEach { it.handleReconciliation(reconDate) }
		holdings.forEach { it.handleEndOfMonth(reconDate) }
		projectionListeners.forEach { it.onEndOfMonth(reconDate) }

		// projection period (recon period is handled explicitly above)
		val startDate = reconDate + 1
		val endDate = periodicity.endOfPeriod(this.endDate)!!

		for (date in startDate..endDate) {

			projectionListeners.forEach { it.onStartOfMonth(date) }

			holdings.forEach { it.handleLifecycle(date, ProjectionPhase.LIFECYCLE) }
			holdings.forEach { it.handleLifecycle(date, ProjectionPhase.EOP_TRANSFER) }
			if (date.isEndOfYear) {
				holdings.filterIsInstance<TaxHandler>().forEach { it.handleTaxes(date, this) }
				val totalProfit = max(household.internalCash.getBalance() - (household.targetCashBalance ?: 0.0), 0.0)
				ValueProviders.setTotalProfit(totalProfit)
				holdings.forEach {
					val availableProfit = max(household.internalCash.getBalance() - (household.targetCashBalance ?: 0.0), 0.0)
					ValueProviders.setAvailableProfit(availableProfit)
					it.handleLifecycle(date, ProjectionPhase.EOY_REBALANCE)
				}
				if (household.targetCashBalance != null) this.rebalance(date)
			}
			holdings.forEach { it.handleEndOfMonth(date) }

			projectionListeners.forEach { it.onEndOfMonth(date) }

		}

		// showProjectionResult(projectionPeriods)

	}

	override fun getProjection(): List<ProjectionPeriod> {
		return projectionPeriods
	}

	fun addProjectionListener(listener: ProjectionListener) {
		projectionListeners.add(listener)
	}

	override fun onStartOfMonth(date: SimDate) {
		// either current month (when monthly) or end of year (when yearly)
		val endOfPeriod = this.periodicity.endOfPeriod(date)!!
		val projectionPeriod = this.projectionPeriod
		if (projectionPeriod == null) {
			this.projectionPeriod = openFirstPeriod(holdings, endOfPeriod)
		} else if (periodicity.isStartOfPeriod(date)) {
			this.projectionPeriod = openNextPeriod(projectionPeriod, endOfPeriod)
		}
	}

	private fun openFirstPeriod(holdings: List<Holding>, date: SimDate): ProjectionPeriod {
		return ProjectionPeriod(
			date = date,
			bookings = mutableListOf(),
			holdings = holdings.map { HoldingPeriod(date, it) }.associateBy { it.holding },
			taxInfo = null,
		)
	}

	private fun openNextPeriod(projectionPeriod: ProjectionPeriod, date: SimDate): ProjectionPeriod {
		return ProjectionPeriod(
			date = date,
			bookings = mutableListOf(),
			holdings = projectionPeriod.holdings.mapValues { (_, h) ->
				h.copy(
					date = date,
					inflows = 0.0,
					outflows = 0.0,
					gain = 0.0,
					bookings = mutableListOf()
				)
			},
			taxInfo = null,
		)
	}

	override fun onEndOfMonth(date: SimDate) {
		if (periodicity.isEndOfPeriod(date)) {
			closeProjectionPeriod(this.projectionPeriod!!)
			projectionPeriods.add(this.projectionPeriod!!)
		}
	}

	private fun closeProjectionPeriod(projectionPeriod: ProjectionPeriod) {
		for (holdingPeriod in projectionPeriod.holdings.values) {
			val hBookings = groupLifecycleBookings(holdingPeriod.bookings)
			holdingPeriod.bookings.clear()
			holdingPeriod.bookings.addAll(hBookings)
			holdingPeriod.bookings.forEach { booking ->
				val bookingKind = booking.bookingKind
				val bookingType = bookingKind.bookingType
				if (bookingKind == BookingKind.RECONCILE) {
					holdingPeriod.balance = booking.amount
				} else if (bookingType == BookingType.CASHFLOW || bookingType == BookingType.TRANSFER) {
					if (bookingKind == BookingKind.CAPITAL_GAIN) {
						holdingPeriod.gain += booking.amount
					} else if (booking.amount > 0) {
						holdingPeriod.inflows += booking.amount
					} else {
						holdingPeriod.outflows += booking.amount
					}
					holdingPeriod.balance += booking.amount
				}
				projectionPeriod.bookings.add(booking)
			}
		}
	}

	private fun groupLifecycleBookings(bookings: List<Booking>): List<Booking> {
		return bookings
			.filterIsInstance<Booking.Lifecycle>()
			.groupBy { periodicity.endOfPeriod(it.date)!! }
			.flatMap { (date, periodBookings) ->
				periodBookings.groupBy { "${it.holding.tag}:${it.bookingKind.code}:${it.counterHolding.tag}:${it.trigHolding.tag}:${it.trigCompo}" }
					.map { (_, keyBookings) ->
						val keyBooking = keyBookings.first()
						val amount = keyBookings.sumOf { it.amount }
						Booking.Lifecycle(
							keyBooking.holding,
							date,
							keyBooking.bookingKind,
							amount,
							keyBooking.counterHolding,
							keyBooking.trigHolding,
							keyBooking.trigCompo
						)
					}
			} + bookings.filterIsInstance<Booking.Reconciliation>()
	}

	override fun onBooking(booking: Booking) {
		projectionListeners.forEach { if (it != this) it.onBooking(booking) }
		val holdingPeriod = this.projectionPeriod!!.holdings[booking.holding]!!
		holdingPeriod.bookings.add(booking)
	}

	private fun rebalance(date: SimDate) {
		val targetCashBalance = household.targetCashBalance
		if (targetCashBalance == null || targetCashBalance <= 0) {
			return
		}
		val internalCashBalance = household.internalCash.getBalance()
		val bufferBalance = household.bufferInvestment.getBalance()
		if (internalCashBalance > targetCashBalance) {
			val amount = internalCashBalance - targetCashBalance
			household.bufferInvestment.bookTransfer(date, amount, household.internalCash, household.bufferInvestment, "rebalance")
		} else if (internalCashBalance < targetCashBalance && bufferBalance > 0) {
			val amount = min(targetCashBalance - internalCashBalance, bufferBalance)
			household.internalCash.bookTransfer(date, amount, household.bufferInvestment, household.bufferInvestment, "rebalance")
		}
	}

	override fun getTaxes(
		taxYear: Int,
		taxLocationId: Int,
		civilStatus: CivilStatus,
		householdInfo: ChTaxHouseholdInfo,
		partner1Info: ChTaxPersonalInfo,
		partner2Info: ChTaxPersonalInfo?,
		children: List<ChTaxChild>
	): ChTaxResult {
		val taxResult = engine.taxService.getTaxes(
			taxYear,
			taxLocationId,
			civilStatus,
			householdInfo,
			partner1Info,
			partner2Info,
			children
		)
		projectionPeriod!!.taxInfo = TaxInfo(
			year = taxYear,
			taxLocationId = taxLocationId,
			civilStatus = civilStatus,
			householdInfo = householdInfo,
			partner1Info = partner1Info,
			partner2Info = partner2Info,
			children = children,
			taxResult = taxResult,
		)
		return taxResult
	}

	private fun showProjectionResult(projectionResult: List<ProjectionPeriod>) {

		println("\nPROJECTION RESULT from ${projectionResult.first().date} to ${projectionResult.last().date}")

		println("\nHolding Aggregates:")
		println("-------------------")
		var lastYear = projectionResult.first().date.year
		for (period in projectionResult) {
			if (period.date.year != lastYear) {
				println()
				lastYear = period.date.year
			}
			for (holding in period.holdings.values) {
				if (holding.holding is Valuable) {
					println(holding)
				}
			}
		}

		println("\nBookings:")
		println("---------")
		lastYear = projectionResult.first().date.year
		for (period in projectionResult) {
			if (period.date.year != lastYear) {
				println()
				lastYear = period.date.year
			}
			for (booking in period.bookings) {
				if (booking is Booking.Reconciliation) {
					println(booking)
				} else if (booking is Booking.Lifecycle) {
					if (booking.holding != household.externalCash) {
						println(booking)
					} else if (booking.counterHolding != household.internalCash) {
						println(booking)
					}
				}
			}
		}

	}

}