package io.fincast.engine

import io.fincast.enums.BookingKind
import io.fincast.holding.Holding

/**
 * A half-booking on a holding, either from reconciliation or triggered by a compo component lifecycle.
 */
sealed interface Booking {

	val holding: Holding
	val date: SimDate
	val bookingKind: BookingKind
	val amount: Double

	/**
	 * A reconciliation half-booking on a holding.
	 * Reconciliation bookings do not have a counter booking.
	 * A reconciliation booking sets the balance of a holding at the given date.
	 */
	data class Reconciliation(
		override val holding: Holding,
		override val date: SimDate,
		override val amount: Double,
	) : Booking {

		override val bookingKind = BookingKind.RECONCILE

		override fun toString(): String {
			return "${date}: ${bookingKind.code} ${holding.tag} ${String.format("%.0f", amount)}"
		}

	}

	/**
	 * A half-booking on a holding, triggered by a holding lifecycle.
	 * This booking is part of a pair of bookings, the other one being on the counter compo with opposite amount.
	 */
	data class Lifecycle(
		override val holding: Holding,
		override val date: SimDate,
		override val bookingKind: BookingKind,
		override val amount: Double,
		val counterHolding: Holding,
		val trigHolding: Holding,
		val trigCompo: String,
	) : Booking {

		override fun toString(): String {
			return "${date}: ${bookingKind.code} ${holding.tag} ${String.format("%.0f", amount)} [against ${counterHolding.tag}, trig by ${trigHolding.tag}.${trigCompo}]"
		}

	}

}
