package io.fincast.compo.impl

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProvider
import io.fincast.engine.SimDate
import io.fincast.enums.BookingKind
import io.fincast.enums.FundsUtilisation
import io.fincast.enums.Periodicity
import io.fincast.holding.Holding

enum class FlowDirection {
	INCOMING,
	OUTGOING
}

/**
 * A cashflow component generates external cashflows (incoming or outgoing).
 */
data class CashflowCompo(
	override val holding: Holding,
	override val tag: String,
	val fundsUtilisation: FundsUtilisation,
	val amount: ValueProvider<Double>,
	val direction: FlowDirection,
	val periodicity: Periodicity,
	val startDate: SimDate? = null,
	val endDate: SimDate? = null,
	val proRata: Boolean = true,
) : FinancialCompo {

	override fun handleLifecycle(date: SimDate) {
		if (periodicity.isEndOfPeriod(date, startDate, endDate)) {
			val sign = if (direction == FlowDirection.INCOMING) 1 else -1
			var amount = (amount.getValue(this, date) ?: 0.0) * sign
			val sop = periodicity.startOfPeriod(date)
			if (proRata && startDate != null && periodicity.months > 1 && sop!! < startDate) {
				amount = amount * (date - startDate + 1) / periodicity.months
			}
			val holding = if (fundsUtilisation == FundsUtilisation.COMPOUND) this.holding else household.internalCash
			holding.bookCashflow(date, BookingKind.CASHFLOW, amount, this.holding, this.tag)
		}
	}

}
