package io.fincast.portfolio.impl

import io.fincast.enums.Periodicity
import io.fincast.portfolio.*

class PortfolioImpl(
	override val reconDate: SimDate,
) : Portfolio {

	private var _pocketMoney: Position? = null
	private var _externalMoney: Position? = null
	override var positions: List<Position> = emptyList()

	private val bookings: MutableList<Booking> = mutableListOf()

	override val pocketMoney: Position
		get() = _pocketMoney ?: throw IllegalStateException("pocketMoney not initialized")

	override val externalMoney: Position
		get() = _externalMoney ?: throw IllegalStateException("externalMoney not initialized")

	override fun initAuxPositions(pocketMoney: Position, externalMoney: Position) {
		_pocketMoney = pocketMoney
		_externalMoney = externalMoney
	}

	override fun initPositions(positions: List<Position>) {
		this.positions = positions
	}

	override fun getPositions(tag: String): List<Position> {
		return positions.filter { it.tag == tag }
	}

	override fun calcProjection(endDate: SimDate, periodicity: Periodicity): List<Booking> {

		check(periodicity == Periodicity.MONTHLY || periodicity == Periodicity.YEARLY) { "periodicity must be MONTHLY or YEARLY" }

		bookings.clear()

		// close balance at reconDate
		bookBalance(reconDate)

		// don't calc reconDate, it is already reconciled
		for (d in reconDate + 1..periodicity.endOfPeriod(endDate)) {
			// Period Lifecycle: Accrual, Dividends, ...
			positions.forEach { it.handlePositionLifecycle(d) }
			// End of Period: Transfers
			positions.forEach { it.handlePortfolioLifecycle(d) }
			// End of Period: Balance Bookings
			bookBalance(d)
		}

		if (periodicity == Periodicity.MONTHLY) {
			return bookings.filter { it is LifecycleBooking || it is AggregateBooking }
		}

		// aggregate bookings at eop
		val reconBookings = bookings.filterIsInstance<ReconciliationBooking>()
		val eopAggregateBookings = bookings
			.filterIsInstance<AggregateBooking>()
			.filter { it.date == reconDate || periodicity.isCashflowDate(it.date) }
		val eopLifecycleBookings = bookings
			.filterIsInstance<LifecycleBooking>()
			.groupBy { periodicity.endOfPeriod(it.date) }
			.flatMap { (date, periodBookings) ->
				periodBookings.groupBy { "${it.position.tag}:${it.refPosition.tag}:${it.refCompo.tag}:${it.bookingKind.code}" }
					.map { (key, keyBookings) ->
						val keyBooking = keyBookings.first()
						val amount = keyBookings.sumOf { it.amount }
						LifecycleBooking(keyBooking.position, date, keyBooking.bookingKind, amount, keyBooking.refPosition, keyBooking.refCompo)
					}
			}
		val eopBookings = (eopAggregateBookings + eopLifecycleBookings).sortedWith(compareBy({ it.date }, { if (it is AggregateBooking) 1 else 0 }))
		return reconBookings + eopBookings

	}

	private fun bookBalance(d: SimDate) {
		positions.forEach { it.bookBalance(d) }
		pocketMoney.bookBalance(d)
		externalMoney.bookBalance(d)
	}

	override fun addBooking(booking: Booking) {
		bookings.add(booking)
	}

}
