package io.fincast.household.impl

import io.fincast.enums.Periodicity
import io.fincast.household.*
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.SimDate
import io.fincast.portfolio.impl.PortfolioImpl

const val POCKET_MONEY_TAG = "#pocketMoney"
const val EXTERNAL_MONEY_TAG = "#externalMoney"

class HouseholdImpl(
	override val persons: List<Person>,
	override val reconDate: SimDate,
	override val holdings: List<Holding>
) : Household {

	override val pocketMoney: Holding
	override val externalMoney: Holding

	override val portfolio: Portfolio

	private val holdingsByTag: Map<String, Holding>

	init {
		pocketMoney = MoneyAccount(POCKET_MONEY_TAG)
		externalMoney = MoneyAccount(EXTERNAL_MONEY_TAG)
		val allHoldings = holdings + pocketMoney + externalMoney
		holdingsByTag = allHoldings.associateBy { it.tag }
		portfolio = createPortfolio()
	}

	override fun calcProjection(endDate: SimDate, periodicity: Periodicity): List<Booking> {
		check(periodicity == Periodicity.MONTHLY || periodicity == Periodicity.YEARLY) { "periodicity must be MONTHLY or YEARLY" }
		return portfolio.calcProjection(endDate, periodicity).map { fromBooking(it) }
	}

	private fun createPortfolio(): Portfolio {
		val portfolio: Portfolio = PortfolioImpl(reconDate)
		val pocketMoney = this.pocketMoney.createPosition(portfolio)
		val externalMoney = this.externalMoney.createPosition(portfolio)
		portfolio.initAuxPositions(pocketMoney, externalMoney)
		val positions = holdings.map { it.createPosition(portfolio) }
		portfolio.initPositions(positions)
		return portfolio
	}

	private fun fromBooking(booking: io.fincast.portfolio.Booking): Booking {
		return when (booking) {
			is io.fincast.portfolio.ReconciliationBooking -> ReconciliationBooking(
				holding = holdingsByTag[booking.position.tag]!!,
				date = booking.date,
				amount = booking.amount,
			)

			is io.fincast.portfolio.LifecycleBooking -> LifecycleBooking(
				holding = holdingsByTag[booking.position.tag]!!,
				date = booking.date,
				amount = booking.amount,
				bookingKind = booking.bookingKind,
				refHolding = booking.refPosition.let { holdingsByTag[it.tag]!! },
				refCompo = booking.refCompo.tag,
			)

			is io.fincast.portfolio.AggregateBooking -> AggregateBooking(
				holding = holdingsByTag[booking.position.tag]!!,
				date = booking.date,
				amount = booking.amount,
				yield = booking.yield,
			)
		}
	}

}
