package io.fincast.household.impl

import io.fincast.enums.FundsAllocation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.household.Contract
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.Position
import io.fincast.portfolio.PositionCompo
import io.fincast.portfolio.SimDate
import io.fincast.portfolio.ValueProviders.constValue
import io.fincast.portfolio.impl.CashflowCompo
import io.fincast.portfolio.impl.TransferCompo

data class ChPillarTwo(
	override val tag: String,
	val reconBalance: Double = 0.0,
	val retirementDate: SimDate,
	val vestedBenefits: Double = 0.0,
	val capitalWithdrawal: Double = 0.0,
	val conversionRate: Double = 0.0,
) : Contract {

	override val productType = ProductType.INCOME
	override val startDate = retirementDate
	override val endDate = null

	class Builder {
		private var tag: String? = null
		private var reconBalance: Double? = null
		private var retirementDate: SimDate? = null
		private var capitalWithdrawal: Double? = null
		private var vestedBenefits: Double? = null
		private var conversionRate: Double? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun reconBalance(reconBalance: Double) = apply { this.reconBalance = reconBalance }
		fun retirementDate(retirementDate: SimDate) = apply { this.retirementDate = retirementDate }
		fun capitalWithdrawal(capitalWithdrawal: Double) = apply { this.capitalWithdrawal = capitalWithdrawal }
		fun vestedBenefits(vestedBenefits: Double) = apply { this.vestedBenefits = vestedBenefits }
		fun conversionRate(conversionRate: Double) = apply { this.conversionRate = conversionRate }

		fun build(): ChPillarTwo {
			return ChPillarTwo(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				retirementDate = retirementDate ?: throw IllegalArgumentException("retirementDate is required"),
				capitalWithdrawal = capitalWithdrawal ?: 0.0,
				vestedBenefits = vestedBenefits ?: 0.0,
				conversionRate = conversionRate ?: 0.0,
			)
		}
	}

	override fun createPosition(portfolio: Portfolio): Position {
		val pos = super.createPosition(portfolio)
		if (reconBalance != 0.0) {
			pos.bookReconciliation(portfolio.reconDate, reconBalance)
		}
		return pos
	}

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		if (vestedBenefits == 0.0) {
			return emptyList()
		}
		val compos: MutableList<PositionCompo> = mutableListOf()
		val capitalContribution = vestedBenefits - reconBalance
		val monthlyContribution = capitalContribution / (retirementDate - portfolio.reconDate)
		compos.add(
			CashflowCompo(
				position = pos,
				tag = "capitalContribution",
				amount = constValue(monthlyContribution),
				sign = 1,
				endDate = retirementDate,
				periodicity = Periodicity.MONTHLY,
				fundsAllocation = FundsAllocation.COMPOUND,
			)
		)
		val capital = if (capitalWithdrawal <= 100.0) (capitalWithdrawal / 100.0 * vestedBenefits) else capitalWithdrawal
		val pensionBase = vestedBenefits - capital
		if (capital != 0.0) {
			compos.add(
				TransferCompo(
					position = pos,
					tag = "capitalWithdrawal",
					fromPosition = pos,
					toPosition = portfolio.pocketMoney,
					amount = constValue(capital),
					sign = -1,
					endDate = retirementDate + 1,
					periodicity = Periodicity.MATURITY,
				)
			)
		}
		if (pensionBase != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "pension",
					fundsAllocation = FundsAllocation.DISBURSE,
					amount = constValue(pensionBase * conversionRate / 100.0 / 12),
					sign = 1,
					startDate = retirementDate + 1,
					periodicity = Periodicity.MONTHLY,
				)
			)
		}
		return compos
	}

}
