package io.fincast.portfolio

import io.fincast.enums.BookingKind
import io.fincast.util.SimDate

sealed interface PositionBooking {

	val position: Position
	val date: SimDate
	val bookingKind: BookingKind
	val amount: Double

	fun isRelevantFor(positionTag: String?): Boolean = (positionTag == null) || position.tag == positionTag

}

/**
 * A reconciliation half-booking on a position.
 *
 * A reconciliation booking sets the balance of a position at the given date.
 */
data class PositionReconciliationBooking(
	override val position: Position,
	override val date: SimDate,
	override val amount: Double,
) : PositionBooking {

	override val bookingKind = BookingKind.RECONCILE

	override fun toString(): String {
		return "${date}: ${bookingKind.code} ${position.tag} ${String.format("%.0f", amount)}"
	}

}

/**
 * A half-booking on a position, triggered by a position component lifecycle.
 */
data class PositionLifecycleBooking(
	override val position: Position,
	override val date: SimDate,
	override val bookingKind: BookingKind,
	override val amount: Double,
	val refPosition: Position,
	val refCompo: PositionCompo,
) : PositionBooking {

	override fun isRelevantFor(positionTag: String?): Boolean = super.isRelevantFor(positionTag) || refPosition.tag == positionTag

	override fun toString(): String {
		return "${date}: ${bookingKind.code} ${position.tag} ${String.format("%.0f", amount)} (${position.tag}:${refCompo.tag})"
	}

}
