package io.fincast.household.impl

import io.fincast.enums.FundsAllocation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.household.Person
import io.fincast.household.Valuable
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.Position
import io.fincast.portfolio.PositionCompo
import io.fincast.portfolio.ValueProviders.constValue
import io.fincast.portfolio.impl.CashflowCompo

data class RealEstate(
	override val tag: String,
	override val owner: Person? = null,
	override val reconBalance: Double = 0.0, // market value
	val fiscalValue: Double? = null,
	val isSelfOccupied: Boolean? = null,
	val imputedRentalValue: Double? = null,
	val rentalIncome: Double? = null,
	val maintenanceRate: Double? = null,
	val maintenancePeriodicity: Periodicity = Periodicity.YEARLY,
) : Valuable {

	override val productType = ProductType.ASSET

	class Builder {
		private var tag: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var fiscalValue: Double? = null
		private var isSelfOccupied: Boolean? = null
		private var imputedRentalValue: Double? = null
		private var rentalIncome: Double? = null
		private var maintenanceRate: Double? = null
		private var maintenancePeriodicity: Periodicity? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun fiscalValue(fiscalValue: Double?) = apply { this.fiscalValue = fiscalValue }
		fun isSelfOccupied(isSelfOccupied: Boolean?) = apply { this.isSelfOccupied = isSelfOccupied }
		fun imputedRentalValue(imputedRentalValue: Double?) = apply { this.imputedRentalValue = imputedRentalValue }
		fun rentalIncome(rentalIncome: Double?) = apply { this.rentalIncome = rentalIncome }
		fun maintenanceRate(maintenanceRate: Double?) = apply { this.maintenanceRate = maintenanceRate }
		fun maintenancePeriodicity(maintenancePeriodicity: Periodicity?) = apply { this.maintenancePeriodicity = maintenancePeriodicity }
		fun build(): RealEstate {
			return RealEstate(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				owner = owner,
				reconBalance = reconBalance ?: 0.0,
				fiscalValue = fiscalValue,
				isSelfOccupied = isSelfOccupied,
				imputedRentalValue = imputedRentalValue,
				rentalIncome = rentalIncome,
				maintenanceRate = maintenanceRate,
				maintenancePeriodicity = maintenancePeriodicity ?: Periodicity.YEARLY,
			)
		}
	}

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		val compos: MutableList<PositionCompo> = mutableListOf()
		if (maintenanceRate != null && maintenanceRate != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "maintenance",
					fundsAllocation = FundsAllocation.DISBURSE,
					amount = constValue(maintenanceRate / 100.0 * reconBalance * maintenancePeriodicity.months / 12),
					sign = -1,
					periodicity = maintenancePeriodicity,
				)
			)
		}
		if (rentalIncome != null && rentalIncome != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "rentalIncome",
					fundsAllocation = FundsAllocation.DISBURSE,
					amount = constValue(rentalIncome / 12),
					sign = 1,
					periodicity = Periodicity.MONTHLY,
				)
			)
		}
		return compos
	}

}
