package io.fincast.household.impl

import io.fincast.enums.FundsAllocation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.household.Person
import io.fincast.household.Valuable
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.Position
import io.fincast.portfolio.PositionCompo
import io.fincast.portfolio.ValueProviders.constValue
import io.fincast.portfolio.ValueProviders.prevEomCreditBalance
import io.fincast.portfolio.ValueProviders.prevEomDebitBalance
import io.fincast.portfolio.impl.InterestCompo

data class MoneyAccount(
	override val tag: String,
	override val owner: Person? = null,
	override val reconBalance: Double = 0.0,
	val creditInterestRate: Double? = 0.0,
	val debitInterestRate: Double? = 0.0,
	val interestPeriodicity: Periodicity = Periodicity.YEARLY,
) : Valuable {

	override val productType = ProductType.VALUABLE

	class Builder {
		private var tag: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var creditInterestRate: Double? = null
		private var debitInterestRate: Double? = null
		private var interestPeriodicity: Periodicity? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun creditInterestRate(creditInterestRate: Double?) = apply { this.creditInterestRate = creditInterestRate }
		fun debitInterestRate(debitInterestRate: Double?) = apply { this.debitInterestRate = debitInterestRate }
		fun interestPeriodicity(interestPeriodicity: Periodicity?) = apply { this.interestPeriodicity = interestPeriodicity }
		fun build(): MoneyAccount {
			return MoneyAccount(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				owner = owner,
				reconBalance = reconBalance ?: 0.0,
				creditInterestRate = creditInterestRate ?: 0.0,
				debitInterestRate = debitInterestRate ?: 0.0,
				interestPeriodicity = interestPeriodicity ?: Periodicity.YEARLY,
			)
		}
	}

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		val compos = mutableListOf<PositionCompo>()
		if (creditInterestRate != null && creditInterestRate != 0.0) {
			compos.add(
				InterestCompo(
					position = pos,
					tag = "creditInterest",
					amount = prevEomCreditBalance(),
					interestRate = constValue(creditInterestRate),
					interestPeriodicity = interestPeriodicity,
					fundsAllocation = FundsAllocation.COMPOUND,
				)
			)
		}
		if (debitInterestRate != null && debitInterestRate != 0.0) {
			compos.add(
				InterestCompo(
					position = pos,
					tag = "debitInterest",
					amount = prevEomDebitBalance(),
					interestRate = constValue(debitInterestRate),
					interestPeriodicity = interestPeriodicity,
					fundsAllocation = FundsAllocation.COMPOUND,
				)
			)
		}
		return compos
	}

}
