package io.fincast.household.impl

import io.fincast.enums.FundsAllocation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.household.Person
import io.fincast.household.Valuable
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.Position
import io.fincast.portfolio.PositionCompo
import io.fincast.portfolio.ValueProvider
import io.fincast.portfolio.ValueProviders.constValue
import io.fincast.portfolio.impl.*
import io.fincast.util.SimDate

data class InvestmentAccount(
	override val tag: String,
	override val owner: Person? = null,
	override val reconBalance: Double = 0.0,

	val contributionAmount: ValueProvider<Double>? = null,
	val contributionPeriodicity: Periodicity = Periodicity.YEARLY,
	val contributionStartDate: SimDate? = null,
	val contributionEndDate: SimDate? = null,

	val cashPart: Double = 0.0,
	val cashInterestRate: Double? = null,
	val bondPart: Double = 0.0,
	val bondInterestRate: Double? = null,
	val interestAllocation: FundsAllocation = FundsAllocation.COMPOUND,

	val stockPart: Double = 0.0,
	val stockCapitalGain: Double? = null,
	val stockDividendYield: Double? = null,
	val dividendAllocation: FundsAllocation = FundsAllocation.COMPOUND,

	val withdrawalAmount: ValueProvider<Double>? = null,
	val withdrawalPeriodicity: Periodicity = Periodicity.YEARLY,
	val withdrawalStartDate: SimDate? = null,
	val withdrawalEndDate: SimDate? = null,

	val managementFee: Double? = 0.0,
	val performanceFee: Double? = 0.0,
) : Valuable {

	override val productType = ProductType.ASSET

	class Builder {
		private var tag: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var contributionAmount: ValueProvider<Double>? = null
		private var contributionPeriodicity: Periodicity? = null
		private var contributionStartDate: SimDate? = null
		private var contributionEndDate: SimDate? = null
		private var cashPart: Double? = null
		private var cashInterestRate: Double? = null
		private var bondPart: Double? = null
		private var bondInterestRate: Double? = null
		private var interestAllocation: FundsAllocation? = null
		private var stockPart: Double? = null
		private var stockCapitalGain: Double? = null
		private var stockDividendYield: Double? = null
		private var dividendAllocation: FundsAllocation? = null
		private var withdrawalAmount: ValueProvider<Double>? = null
		private var withdrawalPeriodicity: Periodicity? = null
		private var withdrawalStartDate: SimDate? = null
		private var withdrawalEndDate: SimDate? = null
		private var managementFee: Double? = null
		private var performanceFee: Double? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun contributionAmount(contributionAmount: ValueProvider<Double>?) = apply { this.contributionAmount = contributionAmount }
		fun contributionPeriodicity(contributionPeriodicity: Periodicity?) = apply { this.contributionPeriodicity = contributionPeriodicity }
		fun contributionStartDate(contributionStartDate: SimDate?) = apply { this.contributionStartDate = contributionStartDate }
		fun contributionEndDate(contributionEndDate: SimDate?) = apply { this.contributionEndDate = contributionEndDate }
		fun cashPart(cashPart: Double?) = apply { this.cashPart = cashPart }
		fun cashInterestRate(cashInterestRate: Double?) = apply { this.cashInterestRate = cashInterestRate }
		fun bondPart(bondPart: Double?) = apply { this.bondPart = bondPart }
		fun bondInterestRate(bondInterestRate: Double?) = apply { this.bondInterestRate = bondInterestRate }
		fun interestAllocation(interestAllocation: FundsAllocation?) = apply { this.interestAllocation = interestAllocation }
		fun stockPart(stockPart: Double?) = apply { this.stockPart = stockPart }
		fun stockCapitalGain(stockCapitalGain: Double?) = apply { this.stockCapitalGain = stockCapitalGain }
		fun stockDividendYield(stockDividendYield: Double?) = apply { this.stockDividendYield = stockDividendYield }
		fun dividendAllocation(dividendAllocation: FundsAllocation?) = apply { this.dividendAllocation = dividendAllocation }
		fun withdrawalAmount(withdrawalAmount: ValueProvider<Double>?) = apply { this.withdrawalAmount = withdrawalAmount }
		fun withdrawalPeriodicity(withdrawalPeriodicity: Periodicity?) = apply { this.withdrawalPeriodicity = withdrawalPeriodicity }
		fun withdrawalStartDate(withdrawalStartDate: SimDate?) = apply { this.withdrawalStartDate = withdrawalStartDate }
		fun withdrawalEndDate(withdrawalEndDate: SimDate?) = apply { this.withdrawalEndDate = withdrawalEndDate }
		fun managementFee(managementFee: Double?) = apply { this.managementFee = managementFee }
		fun performanceFee(performanceFee: Double?) = apply { this.performanceFee = performanceFee }
		fun build(): InvestmentAccount {
			return InvestmentAccount(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				owner = owner,
				reconBalance = reconBalance ?: 0.0,
				//
				contributionAmount = contributionAmount,
				contributionPeriodicity = contributionPeriodicity ?: Periodicity.YEARLY,
				contributionStartDate = contributionStartDate,
				contributionEndDate = contributionEndDate,
				//
				cashPart = cashPart ?: 0.0,
				cashInterestRate = cashInterestRate,
				bondPart = bondPart ?: 0.0,
				bondInterestRate = bondInterestRate,
				interestAllocation = interestAllocation ?: FundsAllocation.COMPOUND,
				stockPart = stockPart ?: 0.0,
				stockCapitalGain = stockCapitalGain,
				stockDividendYield = stockDividendYield,
				dividendAllocation = dividendAllocation ?: FundsAllocation.COMPOUND,
				//
				withdrawalAmount = withdrawalAmount,
				withdrawalPeriodicity = withdrawalPeriodicity ?: Periodicity.YEARLY,
				withdrawalStartDate = withdrawalStartDate,
				withdrawalEndDate = withdrawalEndDate,
				//
				managementFee = managementFee ?: 0.0,
				performanceFee = performanceFee ?: 0.0,
			)
		}
	}

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		val compos: MutableList<PositionCompo> = mutableListOf()
		if (contributionAmount != null) {
			val startDate = SimDate.max(contributionStartDate ?: portfolio.reconDate, portfolio.reconDate)
			compos.add(
				TransferCompo(
					position = pos,
					tag = "contribution",
					fromPosition = portfolio.pocketMoney,
					toPosition = pos,
					amount = contributionAmount,
					sign = -1,
					startDate = startDate,
					endDate = contributionEndDate,
					periodicity = contributionPeriodicity,
				)
			)
		}
		if (cashPart != 0.0) {
			if (cashInterestRate != null && cashInterestRate != 0.0) {
				compos.add(
					InterestCompo(
						position = pos,
						tag = "cashInterest",
						interestRate = constValue(cashPart / 100.0 * cashInterestRate),
						interestPeriodicity = Periodicity.YEARLY,
						fundsAllocation = interestAllocation,
					)
				)
			}
		}
		if (bondPart != 0.0) {
			if (bondInterestRate != null && bondInterestRate != 0.0) {
				compos.add(
					InterestCompo(
						position = pos,
						tag = "bondInterest",
						interestRate = constValue(bondPart / 100.0 * bondInterestRate),
						interestPeriodicity = Periodicity.YEARLY,
						fundsAllocation = interestAllocation,
					)
				)
			}
		}
		if (stockPart != 0.0) {
			if (stockDividendYield != null && stockDividendYield != 0.0) {
				compos.add(
					DividendCompo(
						position = pos,
						dividendYield = constValue(stockPart / 100.0 * stockDividendYield),
						dividendPeriodicity = Periodicity.YEARLY,
						fundsAllocation = dividendAllocation,
					)
				)
			}
			if (stockCapitalGain != null && stockCapitalGain != 0.0) {
				compos.add(
					CapitalGainCompo(
						position = pos,
						capitalGainRate = constValue(stockPart / 100.0 * stockCapitalGain),
					)
				)
			}
		}
		if (performanceFee != null && performanceFee != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "performanceFee",
					fundsAllocation = FundsAllocation.COMPOUND,
					amount = ValueProvider { _, date ->
						// TODO: handle case for positions in the future, with reconBalance == 0.0 at reconDate
						val sop = if ((date - 12) < portfolio.reconDate) {
							portfolio.reconDate
						} else {
							date - 12
						}
						val delta = pos.getBalance(date) - pos.getBalance(sop)
						return@ValueProvider performanceFee / 100 * delta
					},
					sign = -1,
					periodicity = Periodicity.YEARLY,
					startDate = portfolio.reconDate,
				)
			)
		}
		if (managementFee != null && managementFee != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "managementFee",
					fundsAllocation = FundsAllocation.COMPOUND,
					amount = ValueProvider { compo, date -> return@ValueProvider managementFee / 100.0 * compo.position.getBalance(date - 1) },
					sign = -1,
					periodicity = Periodicity.YEARLY,
					startDate = portfolio.reconDate,
				)
			)
		}
		if (withdrawalAmount != null) {
			val startDate = SimDate.max(withdrawalStartDate ?: portfolio.reconDate, portfolio.reconDate)
			compos.add(
				TransferCompo(
					position = pos,
					tag = "withdrawal",
					fromPosition = pos,
					toPosition = portfolio.pocketMoney,
					amount = withdrawalAmount,
					sign = -1,
					startDate = startDate,
					endDate = withdrawalEndDate,
					periodicity = withdrawalPeriodicity,
				)
			)
		}
		return compos
	}

}
