package io.fincast.household.impl

import io.fincast.enums.FundsAllocation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.household.Contract
import io.fincast.household.Person
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.Position
import io.fincast.portfolio.PositionCompo
import io.fincast.portfolio.ValueProvider
import io.fincast.portfolio.impl.CashflowCompo
import io.fincast.util.SimDate

/**
 * Income
 *
 * Income generates an incoming cashflow (external to pocket) with a given periodicity from startDate to endDate.
 *
 * The income can be attached to a person, in which case the income is bound by the person's retirement date, if no endDate is given.
 *
 * @param tag unique identifier of the holding
 * @param owner the (optional) owner of the holding
 * @param amount the amount of income at a given date
 * @param periodicity the periodicity of the income
 * @param startDate the (optional) start date of the income, if omitted will start at the recon date + 1 month
 * @param endDate the (optional) end date of the income, if omitted will be the retirement date of the owner, if any, otherwise will be the end of the simulation
 */
data class Income(
	override val tag: String,
	override val owner: Person? = null,
	val amount: ValueProvider<Double>,
	val periodicity: Periodicity = Periodicity.YEARLY,
	override val startDate: SimDate? = null,
	override val endDate: SimDate? = null,
) : Contract {

	override val productType = ProductType.INCOME

	class Builder {
		private var tag: String? = null
		private var owner: Person? = null
		private var amount: ValueProvider<Double>? = null
		private var periodicity: Periodicity? = null
		private var startDate: SimDate? = null
		private var endDate: SimDate? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun amount(amount: ValueProvider<Double>) = apply { this.amount = amount }
		fun periodicity(periodicity: Periodicity?) = apply { this.periodicity = periodicity }
		fun startDate(startDate: SimDate?) = apply { this.startDate = startDate }
		fun endDate(endDate: SimDate?) = apply { this.endDate = endDate }
		fun build(): Income {
			return Income(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				owner = owner,
				amount = amount ?: throw IllegalArgumentException("amount is required"),
				periodicity = periodicity ?: Periodicity.YEARLY,
				startDate = startDate,
				endDate = endDate,
			)
		}
	}

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		val startDate = SimDate.max(startDate ?: portfolio.reconDate, portfolio.reconDate)
		val endDate = endDate ?: owner?.retirementDate
		val cf = CashflowCompo(
			position = pos,
			tag = "income",
			fundsAllocation = FundsAllocation.DISBURSE,
			amount = amount,
			sign = 1,
			startDate = startDate,
			endDate = endDate,
			periodicity = periodicity,
		)
		return listOf(cf)
	}

}
