package io.fincast.household.impl

import io.fincast.enums.Periodicity
import io.fincast.household.*
import io.fincast.portfolio.*
import io.fincast.portfolio.impl.PortfolioImpl
import io.fincast.util.SimDate

class HouseholdImpl(
	override val partner1: Person,
	override val partner2: Person?,
	override val reconDate: SimDate,
	override val holdings: List<Holding>
) : Household {

	override val pocketMoney: Holding
	override val externalMoney: Holding

	override val portfolio: Portfolio

	private val holdingsByTag: Map<String, Holding>

	init {
		pocketMoney = MoneyAccount(Household.POCKET_MONEY_TAG)
		externalMoney = MoneyAccount(Household.EXTERNAL_MONEY_TAG)
		val allHoldings = holdings + pocketMoney + externalMoney
		holdingsByTag = allHoldings.associateBy { it.tag }
		portfolio = createPortfolio()
	}

	override fun calcProjection(endDate: SimDate, periodicity: Periodicity): ProjectionResult {
		check(periodicity == Periodicity.MONTHLY || periodicity == Periodicity.YEARLY) { "periodicity must be MONTHLY or YEARLY" }
		val portfolioProjectionResult = portfolio.calcProjection(endDate, periodicity)
		return ProjectionResult(
			reconDate = portfolioProjectionResult.reconDate,
			endDate = portfolioProjectionResult.endDate,
			periodicity = portfolioProjectionResult.periodicity,
			holdings = holdings,
			pocketMoney = pocketMoney,
			externalMoney = externalMoney,
			householdAggregates = portfolioProjectionResult.portfolioAggregates.map { fromAggregate(it) },
			holdingAggregates = portfolioProjectionResult.positionAggregates.map { fromAggregate(it) },
			bookings = portfolioProjectionResult.bookings.map { fromBooking(it) },
		)
	}

	override fun calcProjection(holding: Holding, endDate: SimDate, periodicity: Periodicity): ProjectionResult {
		check(periodicity == Periodicity.MONTHLY || periodicity == Periodicity.YEARLY) { "periodicity must be MONTHLY or YEARLY" }
		val portfolioProjectionResult = portfolio.calcProjection(holding.tag, endDate, periodicity)
		return ProjectionResult(
			reconDate = portfolioProjectionResult.reconDate,
			endDate = portfolioProjectionResult.endDate,
			periodicity = portfolioProjectionResult.periodicity,
			holdings = listOf(holding),
			pocketMoney = pocketMoney,
			externalMoney = externalMoney,
			householdAggregates = portfolioProjectionResult.portfolioAggregates.map { fromAggregate(it) },
			holdingAggregates = portfolioProjectionResult.positionAggregates.map { fromAggregate(it) },
			bookings = portfolioProjectionResult.bookings.map { fromBooking(it) },
		)
	}

	private fun createPortfolio(): Portfolio {
		val portfolio: Portfolio = PortfolioImpl(reconDate)
		val pocketMoney = this.pocketMoney.createPosition(portfolio)
		val externalMoney = this.externalMoney.createPosition(portfolio)
		portfolio.initAuxPositions(pocketMoney, externalMoney)
		val positions = holdings.map { it.createPosition(portfolio) }
		portfolio.initPositions(positions)
		return portfolio
	}

	// TODO: aggregate positions to holdings
	private fun fromAggregate(aggregate: PositionAggregate): HoldingAggregate {
		return HoldingAggregate(
			holding = holdingsByTag[aggregate.position.tag]!!,
			date = aggregate.date,
			balance = aggregate.balance,
			yield = aggregate.yield,
			inflows = aggregate.inflows,
			outflows = aggregate.outflows,
			gain = aggregate.gain,
		)
	}

	private fun fromAggregate(aggregate: PortfolioAggregate): HouseholdAggregate {
		return HouseholdAggregate(
			date = aggregate.date,
			assets = aggregate.assets,
			liabilities = aggregate.liabilities,
			yield = aggregate.yield,
			inflows = aggregate.inflows,
			outflows = aggregate.outflows,
			gain = aggregate.gain,
		)
	}

	private fun fromBooking(booking: PositionBooking): Booking {
		return when (booking) {
			is PositionReconciliationBooking -> ReconciliationBooking(
				holding = holdingsByTag[booking.position.tag]!!,
				date = booking.date,
				amount = booking.amount,
			)

			is PositionLifecycleBooking -> LifecycleBooking(
				holding = holdingsByTag[booking.position.tag]!!,
				date = booking.date,
				amount = booking.amount,
				bookingKind = booking.bookingKind,
				refHolding = booking.refPosition.let { holdingsByTag[it.tag]!! },
				refCompo = booking.refCompo.tag,
			)
		}
	}

}
