package io.fincast.household.impl

import io.fincast.enums.FundsAllocation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.household.Person
import io.fincast.household.Valuable
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.Position
import io.fincast.portfolio.PositionCompo
import io.fincast.portfolio.ValueProviders.constValue
import io.fincast.portfolio.impl.CashflowCompo
import io.fincast.portfolio.impl.TransferCompo

data class ChPillarTwo(
	override val tag: String,
	override val owner: Person,
	override val reconBalance: Double = 0.0,
	val projectedVestedBenefits: Double = 0.0,
	val capitalWithdrawal: Double = 0.0,
	val conversionRate: Double = 0.0,
) : Valuable {

	override val productType = ProductType.ASSET

	class Builder {
		private var tag: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var projectedVestedBenefits: Double? = null
		private var capitalWithdrawal: Double? = null
		private var conversionRate: Double? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun owner(owner: Person) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun projectedVestedBenefits(vestedBenefits: Double?) = apply { this.projectedVestedBenefits = vestedBenefits }
		fun capitalWithdrawal(capitalWithdrawal: Double?) = apply { this.capitalWithdrawal = capitalWithdrawal }
		fun conversionRate(conversionRate: Double?) = apply { this.conversionRate = conversionRate }

		fun build(): ChPillarTwo {
			return ChPillarTwo(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				owner = owner ?: throw IllegalArgumentException("owner is required"),
				projectedVestedBenefits = projectedVestedBenefits ?: 0.0,
				capitalWithdrawal = capitalWithdrawal ?: 0.0,
				conversionRate = conversionRate ?: 0.0,
			)
		}
	}

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		if (projectedVestedBenefits == 0.0) {
			return emptyList()
		}
		val compos: MutableList<PositionCompo> = mutableListOf()
		val capitalContribution = projectedVestedBenefits - reconBalance
		val monthlyContribution = capitalContribution / (owner.retirementDate - portfolio.reconDate)
		compos.add(
			CashflowCompo(
				position = pos,
				tag = "capitalContribution",
				amount = constValue(monthlyContribution),
				sign = 1,
				endDate = owner.retirementDate,
				periodicity = Periodicity.MONTHLY,
				fundsAllocation = FundsAllocation.COMPOUND,
			)
		)
		val capital = if (capitalWithdrawal <= 100.0) (capitalWithdrawal / 100.0 * projectedVestedBenefits) else capitalWithdrawal
		if (capital != 0.0) {
			compos.add(
				TransferCompo(
					position = pos,
					tag = "capitalWithdrawal",
					fromPosition = pos,
					toPosition = portfolio.pocketMoney,
					amount = constValue(capital),
					sign = -1,
					endDate = owner.retirementDate + 1,
					periodicity = Periodicity.MATURITY,
				)
			)
		}
		val pensionBase = projectedVestedBenefits - capital
		if (pensionBase != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "pension",
					fundsAllocation = FundsAllocation.DISBURSE,
					amount = constValue(pensionBase * conversionRate / 100.0 / 12),
					sign = 1,
					startDate = owner.retirementDate + 1,
					periodicity = Periodicity.MONTHLY,
				)
			)
		}
		return compos
	}

}
