package io.fincast.household.impl

import io.fincast.enums.FundsAllocation
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.household.Person
import io.fincast.household.Valuable
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.Position
import io.fincast.portfolio.PositionCompo
import io.fincast.portfolio.ValueProviders.constValue
import io.fincast.portfolio.impl.CashflowCompo
import io.fincast.portfolio.impl.InterestCompo

data class Asset(
	override val tag: String,
	override val owner: Person? = null,
	override val reconBalance: Double = 0.0,
	val interestRate: Double = 0.0,
	val interestPeriodicity: Periodicity = Periodicity.YEARLY,
	val interestAllocation: FundsAllocation = FundsAllocation.DISBURSE,
	val insuranceRate: Double = 0.0,
	val insurancePeriodicity: Periodicity = Periodicity.YEARLY,
) : Valuable {

	override val productType = ProductType.ASSET

	class Builder {
		private var tag: String? = null
		private var owner: Person? = null
		private var reconBalance: Double? = null
		private var interestRate: Double? = null
		private var interestPeriodicity: Periodicity? = null
		private var interestAllocation: FundsAllocation? = null
		private var insuranceRate: Double? = null
		private var insurancePeriodicity: Periodicity? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun owner(owner: Person) = apply { this.owner = owner }
		fun reconBalance(reconBalance: Double?) = apply { this.reconBalance = reconBalance }
		fun interestRate(interestRate: Double?) = apply { this.interestRate = interestRate }
		fun interestPeriodicity(interestPeriodicity: Periodicity?) = apply { this.interestPeriodicity = interestPeriodicity }
		fun interestAllocation(fundsAllocation: FundsAllocation?) = apply { this.interestAllocation = fundsAllocation }
		fun insuranceRate(insuranceRate: Double?) = apply { this.insuranceRate = insuranceRate }
		fun insurancePeriodicity(insurancePeriodicity: Periodicity?) = apply { this.insurancePeriodicity = insurancePeriodicity }
		fun build(): Asset {
			return Asset(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				owner = owner,
				reconBalance = reconBalance ?: 0.0,
				interestRate = interestRate ?: 0.0,
				interestPeriodicity = interestPeriodicity ?: Periodicity.YEARLY,
				interestAllocation = interestAllocation ?: FundsAllocation.DISBURSE,
				insuranceRate = insuranceRate ?: 0.0,
				insurancePeriodicity = insurancePeriodicity ?: Periodicity.YEARLY,
			)
		}
	}

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		val compos: MutableList<PositionCompo> = mutableListOf()
		if (interestRate != 0.0) {
			compos.add(
				InterestCompo(
					position = pos,
					interestRate = constValue(interestRate),
					interestPeriodicity = Periodicity.YEARLY,
					fundsAllocation = interestAllocation,
				)
			)
		}
		if (insuranceRate != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "insurance",
					fundsAllocation = FundsAllocation.DISBURSE,
					amount = constValue(insuranceRate / 100.0 * reconBalance * insurancePeriodicity.months / 12),
					sign = -1,
					periodicity = insurancePeriodicity,
				)
			)
		}
		return compos
	}

}
