package io.fincast.portfolio.impl

import io.fincast.enums.BookingKind
import io.fincast.enums.BookingType
import io.fincast.enums.ProductType
import io.fincast.portfolio.*
import io.fincast.util.SimDate

class PositionImpl(
	override val portfolio: Portfolio,
	override val tag: String,
	override val productType: ProductType,
) : Position {

	override var compos: List<PositionCompo> = emptyList()

	private val bookings: MutableList<PositionBooking> = mutableListOf()

	private val aggregates: MutableMap<SimDate, PositionAggregate> = mutableMapOf()

	private var reconDate: SimDate? = null

	private var balance = 0.0
	private var inflows = 0.0
	private var outflows = 0.0
	private var gain = 0.0

	override fun handlePositionLifecycle(date: SimDate) {
		compos.forEach { it.handleLifecycle(date) }
	}

	override fun handlePortfolioLifecycle(date: SimDate) {
	}

	override fun isReconciled(date: SimDate): Boolean = reconDate != null && reconDate == date

	override fun getBalance(date: SimDate): Double {
		return getPeriod(date).balance
	}

	override fun getYield(date: SimDate): Double {
		return getPeriod(date).yield
	}

	override fun getTurnover(date: SimDate): Double {
		return getBalance(date) - getBalance(date - 1)
	}

	override fun getBookings(): List<PositionBooking> {
		return bookings
	}

	override fun getBookings(date: SimDate): List<PositionBooking> {
		return bookings.filter { it.date == date }
	}

	override fun getRefBookings(date: SimDate, refPos: Position): List<PositionBooking> {
		return getBookings(date)
			.stream()
			.filter { b -> b is PositionLifecycleBooking }
			.filter { b -> (b as PositionLifecycleBooking).refPosition === refPos }
			.toList()
	}

	override fun getPeriods(): List<PositionAggregate> {
		return aggregates.values.sortedBy { it.date }
	}

	private fun getPeriod(date: SimDate): PositionAggregate {
		if (aggregates.isEmpty() || date < getFirstDate()!!) {
			return PositionAggregate(this, date, 0.0, 0.0, 0.0, 0.0, 0.0)
		}
		return aggregates[date]!!
	}

	override fun bookReconciliation(date: SimDate, balance: Double) {
		addBooking(PositionReconciliationBooking(this, date, balance))
	}

	override fun bookTurnover(date: SimDate, bookingKind: BookingKind, turnover: Double, refCompo: PositionCompo) {
		addBooking(PositionLifecycleBooking(this, date, bookingKind, turnover, refCompo.position, refCompo))
	}

	override fun addAggregate(date: SimDate) {
		if (bookings.isNotEmpty()) {
			val aggregate = PositionAggregate(this, date, balance, getYield(), inflows, outflows, gain)
			aggregates[date] = aggregate
			portfolio.addAggregate(aggregate)
			inflows = 0.0
			outflows = 0.0
			gain = 0.0
		}
	}

	private fun addBooking(booking: PositionBooking) {
		require(booking.date >= (getLastDate() ?: booking.date))
		{ "bookings must be sequential: ${booking.date} >= ${getLastDate()}" }
		require(reconDate == null || booking.date > reconDate!!) { "cannot add booking to reconciled period" }
		if (0.0 == booking.amount) {
			return
		}
		when (booking.bookingKind.bookingType) {
			BookingType.RECONCILE -> {
				val turnover = booking.amount - balance
				if (turnover > 0) {
					inflows += turnover
				} else {
					outflows += turnover
				}
				balance = booking.amount
				reconDate = booking.date
			}

			BookingType.TURNOVER -> {
				if (booking.bookingKind == BookingKind.CAPITAL_GAIN) {
					gain += booking.amount
				} else if (booking.amount > 0) {
					inflows += booking.amount
				} else {
					outflows += booking.amount
				}
				balance += booking.amount
			}
		}
		bookings.add(booking)
		portfolio.addBooking(booking)
	}

	private fun getFirstDate(): SimDate? {
		return if (bookings.isNotEmpty()) bookings[0].date else null
	}

	private fun getLastDate(): SimDate? {
		return if (bookings.isNotEmpty()) bookings[bookings.lastIndex].date else null
	}

}
