package io.fincast.portfolio.impl

import io.fincast.enums.BookingType
import io.fincast.enums.Periodicity
import io.fincast.portfolio.*
import io.fincast.util.SimDate

class PortfolioImpl(
	override val reconDate: SimDate,
) : Portfolio {

	private var _pocketMoney: Position? = null
	private var _externalMoney: Position? = null
	override var positions: List<Position> = emptyList()

	private val bookings: MutableList<PositionBooking> = mutableListOf()
	private val positionAggregates: MutableList<PositionAggregate> = mutableListOf()
	private val portfolioAggregates: MutableList<PortfolioAggregate> = mutableListOf()

	// local aggregate, ignore internal transfers
	private var assets = 0.0
	private var liabilities = 0.0
	private var inflows = 0.0
	private var outflows = 0.0
	private var yield = 0.0
	private var gain = 0.0

	override val pocketMoney: Position
		get() = _pocketMoney ?: throw IllegalStateException("pocketMoney not initialized")

	override val externalMoney: Position
		get() = _externalMoney ?: throw IllegalStateException("externalMoney not initialized")

	override fun initAuxPositions(pocketMoney: Position, externalMoney: Position) {
		_pocketMoney = pocketMoney
		_externalMoney = externalMoney
	}

	override fun initPositions(positions: List<Position>) {
		this.positions = positions
	}

	override fun getPositions(tag: String): List<Position> {
		return positions.filter { it.tag == tag }
	}

	override fun calcProjection(endDate: SimDate, periodicity: Periodicity): PortfolioProjectionResult {

		require(periodicity == Periodicity.MONTHLY || periodicity == Periodicity.YEARLY) { "periodicity must be MONTHLY or YEARLY" }

		// clear and create first aggregates at reconDate
		positionAggregates.clear()
		addAggregates(reconDate)

		// don't calc reconDate, it is already reconciled
		val periodEndDate = periodicity.endOfPeriod(endDate)
		for (d in reconDate + 1..periodEndDate) {
			// Period Lifecycle: Cashflows, Interest, Dividends, ...
			positions.forEach { it.handlePositionLifecycle(d) }
			// End of Period: Rebalancing Transfers
			positions.forEach { it.handlePortfolioLifecycle(d) }
			// End of Period: Aggregation
			addAggregates(d)
		}

		if (periodicity == Periodicity.MONTHLY) {
			return PortfolioProjectionResult(
				reconDate = reconDate,
				endDate = periodEndDate,
				periodicity = periodicity,
				//
				positions = positions,
				pocketMoney = pocketMoney,
				externalMoney = externalMoney,
				//
				portfolioAggregates = portfolioAggregates,
				positionAggregates = positionAggregates,
				bookings = bookings,
			)
		}

		// aggregate aggregates at eop
		val eopPositionAggregates = positionAggregates
			.groupBy { "${it.position.tag}:${periodicity.endOfPeriod(it.date)}" }
			.map { (date, aggregates) ->
				val lastAggregate = aggregates.maxByOrNull { it.date }!!
				val inflows = aggregates.sumOf { it.inflows }
				val outflows = aggregates.sumOf { it.outflows }
				lastAggregate.copy(inflows = inflows, outflows = outflows)
			}

		val eopPortfolioAggregates = portfolioAggregates
			.groupBy { periodicity.endOfPeriod(it.date) }
			.map { (date, aggregates) ->
				val lastAggregate = aggregates.maxByOrNull { it.date }!!
				val inflows = aggregates.sumOf { it.inflows }
				val outflows = aggregates.sumOf { it.outflows }
				lastAggregate.copy(inflows = inflows, outflows = outflows)
			}

		val eopBookings = bookings
			.filterIsInstance<PositionLifecycleBooking>()
			.groupBy { periodicity.endOfPeriod(it.date) }
			.flatMap { (date, periodBookings) ->
				periodBookings.groupBy { "${it.position.tag}:${it.bookingKind.code}:${it.refPosition.tag}:${it.refCompo.tag}" }
					.map { (_, keyBookings) ->
						val keyBooking = keyBookings.first()
						val amount = keyBookings.sumOf { it.amount }
						PositionLifecycleBooking(keyBooking.position, date, keyBooking.bookingKind, amount, keyBooking.refPosition, keyBooking.refCompo)
					}
			}

		return PortfolioProjectionResult(
			reconDate = reconDate,
			endDate = periodEndDate,
			periodicity = periodicity,
			//
			positions = positions,
			pocketMoney = pocketMoney,
			externalMoney = externalMoney,
			//
			portfolioAggregates = eopPortfolioAggregates,
			positionAggregates = eopPositionAggregates,
			bookings = eopBookings,
		)
	}

	private fun addAggregates(d: SimDate) {
		positions.forEach { it.addAggregate(d) }
		pocketMoney.addAggregate(d)
		externalMoney.addAggregate(d)
		this.addAggregate(d)
	}

	// need to calculate inflows and outflows on portfolio level to cancel internal transfers
	override fun addBooking(booking: PositionBooking) {
		bookings.add(booking)
		if (booking.position == externalMoney) {
			return
		}
		when (booking.bookingKind.bookingType) {
			BookingType.RECONCILE -> {
				val turnover = booking.amount - booking.position.getBalance(booking.date - 1)
				if (turnover > 0) {
					inflows += turnover
				} else {
					outflows += turnover
				}
			}

			BookingType.TURNOVER -> {
				if (booking.bookingKind.isCashflow) {
					if (booking.amount > 0) {
						inflows += booking.amount
					} else {
						outflows += booking.amount
					}
				}
			}
		}
	}

	override fun addAggregate(aggregate: PositionAggregate) {
		positionAggregates.add(aggregate)
		if (aggregate.position == externalMoney) {
			return
		}
		if (aggregate.balance > 0) {
			assets += aggregate.balance
		} else {
			liabilities += aggregate.balance
		}
		yield += aggregate.yield
		gain += aggregate.gain
	}

	override fun addAggregate(date: SimDate) {
		portfolioAggregates.add(PortfolioAggregate(date, assets, liabilities, yield, inflows, outflows, gain))
		assets = 0.0
		liabilities = 0.0
		inflows = 0.0
		outflows = 0.0
		yield = 0.0
		gain = 0.0
	}

	private fun printProjectionResult(projectionResult: PortfolioProjectionResult) {

		println("\nPORTFOLIO Projection from ${projectionResult.reconDate} to ${projectionResult.endDate} with periodicity ${projectionResult.periodicity}")

		println("\nPortfolio Aggregates:")
		println("---------------------")
		for (aggregate in projectionResult.portfolioAggregates) {
			println(aggregate)
		}

		println("\nPosition Aggregates:")
		println("--------------------")
		for (aggregate in projectionResult.positionAggregates) {
			println(aggregate)
		}

		println("\nBookings:")
		println("---------")
		for (booking in projectionResult.bookings) {
			println(booking)
		}

		println("\nPosition Bookings:")
		println("------------------")
		for (pos in positions) {
			println("\n${pos.tag} Bookings:")
			println("------------------")
			for (booking in pos.getBookings()) {
				println(booking)
			}
		}

		println("\nPocketMoney Bookings:")
		println("---------------------")
		for (booking in pocketMoney.getBookings()) {
			println(booking)
		}

		println("\nExternalMoney Bookings:")
		println("-----------------------")
		for (booking in externalMoney.getBookings()) {
			println(booking)
		}

	}

}
