package io.fincast.pos.model.impl

import io.fincast.household.model.enums.ProductType
import io.fincast.pos.model.*
import io.fincast.pos.model.enums.BookingKind

private const val POCKET_MONEY_TAG = "#pocketMoney"
private const val EXTERNAL_MONEY_TAG = "#externalMoney"

class PortfolioImpl(
	override val reconDate: SimDate,
) : Portfolio {

	override var positions: List<Position> = emptyList()
	override val pocketMoney: Position = PositionImpl(this, tag = POCKET_MONEY_TAG, ProductType.VALUABLE)
	override val externalMoney: Position = PositionImpl(this, tag = EXTERNAL_MONEY_TAG, ProductType.VALUABLE)

	private val _bookingPeriodsByPos: MutableMap<Position, MutableList<BookingPeriod>> = mutableMapOf()

	override fun getPositions(tag: String): List<Position> {
		return positions.filter { it.tag == tag }
	}

	fun getBalance(pos: Position, date: SimDate): Double {
		val firstDate = getFirstDate(pos)
		return if (null == firstDate || date < firstDate) 0.0 else getBookingPeriod(pos, date).getBalance()
	}

	fun addReconciliation(pos: Position, date: SimDate, balance: Double) {
		this.addBooking(pos, date, BookingKind.RECONCILE, balance)
	}

	fun getTurnover(pos: Position, date: SimDate): Double {
		val period = getBookingPeriod(pos, date)
		return period.getBalance() - pos.getBalance(date - 1)
	}


	fun addTurnover(pos: Position, date: SimDate, turnover: Double) {
		this.addBooking(pos, date, BookingKind.TURNOVER, turnover)
	}

	fun getInterestAccrual(pos: Position, date: SimDate): Double {
		return getAccrual(pos, date, BookingKind.INTEREST)
	}

	fun getDividendAccrual(pos: Position, date: SimDate): Double {
		return getAccrual(pos, date, BookingKind.DIVIDEND)
	}

	private fun getAccrual(pos: Position, date: SimDate, bookingKind: BookingKind): Double {
		// need to go over externalMoney counter bookings
		// because accrual might not be reinvested on this position
		val refBookings = getBookingPeriod(externalMoney, date).getRefBookings(pos)
		return -refBookings
			.stream()
			.filter { b -> b.bookingKind === bookingKind }
			.mapToDouble(Booking::amount)
			.sum()
	}

	fun getCapitalGain(pos: Position, date: SimDate): Double {
		return getBookings(pos, date)
			.stream()
			.filter { b -> BookingKind.CAPITAL_GAIN === b.bookingKind }
			.mapToDouble(Booking::amount)
			.sum()
	}

	fun getBookings(pos: Position, date: SimDate?): List<Booking> {
		if (null != date) {
			return getBookingPeriod(pos, date).getBookings()
		}
		val bookings: MutableList<Booking> = ArrayList()
		for (period in getBookingPeriods(pos)) {
			bookings.addAll(period.getBookings())
		}
		return bookings
	}

	fun addBooking(pos: Position, date: SimDate, bookingKind: BookingKind, amount: Double, refPos: Position? = null) {
		if (0.0 == amount) {
			return
		}
		val lastDate = getLastDate(pos)
		require(null == lastDate || date >= lastDate) { "new bookingDate after lastBookingDate" }
		val booking = Booking(date, bookingKind, amount, refPos)
		this.addBooking(pos, booking)
	}

	private fun addBooking(pos: Position, booking: Booking) {
		val period = getBookingPeriod(pos, booking.date)
		period.addBooking(booking)
	}

	fun getPeriods(pos: Position) = getBookingPeriods(pos)

	private fun getBookingPeriod(pos: Position, date: SimDate): BookingPeriod {
		val firstDate = getFirstDate(pos)
		// return empty booking period if before first booking date
		if (null != firstDate && date < firstDate) {
			return BookingPeriodImpl(pos, date)
		}
		val bookingPeriods = getBookingPeriods(pos)
		val index = date - (firstDate ?: date)
		// add missing periods
		for (i in bookingPeriods.size..index) {
			if (0 == i) { // first period: create empty period
				val period = BookingPeriodImpl(pos, date)
				bookingPeriods.add(period)
			} else { // subsequent period: copy previous period
				val prevPeriod = bookingPeriods[i - 1]
				val periodDate = bookingPeriods[0].date + i
				val nextPeriod = (prevPeriod as BookingPeriodImpl).copy(periodDate)
				check(nextPeriod.getBookings().isEmpty()) { "no bookings" }
				check(!nextPeriod.isReconciled()) { "not reconciled" }
				bookingPeriods.add(nextPeriod)
			}
		}
		return bookingPeriods[index]
	}

	private fun getFirstDate(pos: Position): SimDate? {
		val bookingPeriods = getBookingPeriods(pos)
		return if (bookingPeriods.isNotEmpty()) bookingPeriods[0].date else null
	}

	private fun getLastDate(pos: Position): SimDate? {
		val bookingPeriods = getBookingPeriods(pos)
		return if (bookingPeriods.isNotEmpty()) bookingPeriods[bookingPeriods.size - 1].date else null
	}

	private fun getBookingPeriods(pos: Position): MutableList<BookingPeriod> {
		if (!_bookingPeriodsByPos.containsKey(pos)) {
			_bookingPeriodsByPos[pos] = mutableListOf()
		}
		return _bookingPeriodsByPos[pos]!!
	}

}
