package io.fincast.pos.model.impl

import io.fincast.pos.model.BookingPeriod
import io.fincast.pos.model.SimDate
import io.fincast.pos.model.enums.BookingType
import io.fincast.pos.model.Position
import io.fincast.pos.model.Booking

class BookingPeriodImpl(
	override val position: Position,
	override val date: SimDate,
) : BookingPeriod {

	private var _balance: Double = 0.0

	private val _bookings: MutableList<Booking> = mutableListOf()

	private var _isReconciled: Boolean = false

	override fun isReconciled(): Boolean = _isReconciled

	override fun getBalance(): Double = _balance

	override fun getBookings(): List<Booking> = _bookings.toList()

	override fun getTurnover(): Double {
		return _bookings
			.stream()
			.filter { b -> BookingType.TURNOVER === b.bookingKind.bookingType }
			.mapToDouble(Booking::amount)
			.sum()
	}

	override fun getValue() = if (position.productType.isValuable) this.getBalance() else 0.0

	override fun getRefBookings(refPos: Position): List<Booking> {
		return _bookings
			.stream()
			.filter { b -> b.refPosition === refPos }
			.toList()
	}

	override fun addBooking(booking: Booking) {
		require(!this._isReconciled) { "dont add booking to reconciled period" }
		when (booking.bookingKind.bookingType) {
			BookingType.RECONCILIATION -> {
				this._balance = booking.amount
				this._isReconciled = true
			}
			BookingType.TURNOVER -> {
				this._balance += booking.amount
			}
		}
		_bookings.add(booking)
	}

	fun copy(date: SimDate): BookingPeriodImpl {
		val period = BookingPeriodImpl(this.position, date)
		period._balance = this.getBalance()
		return period
	}

	override fun toString(): String {
		val turnover: Double = this.getTurnover()
		return "BookingPeriod($date): ${String.format("%.0f", this.getBalance())} [turnover: ${String.format("%.0f", turnover)}${if (_bookings.isNotEmpty()) ", Bookings: $_bookings" else ""}]"
	}

}
