package io.fincast.pos.model

import java.time.LocalDate

/**
 * A point in time with monthly resolution.
 */
class SimDate internal constructor(private val date: Int) : Comparable<SimDate> {

	/**
	 * Year of simulation date.
	 *
	 * @return year
	 */
	fun year() = date / MONTH_PER_YEAR

	/**
	 * Month of simulation date (1 .. 12)
	 *
	 * @return month
	 */
	fun month() = date % MONTH_PER_YEAR + 1

	/**
	 * Add number of months to given [SimDate], return corresponding
	 * [SimDate]
	 *
	 * @param monthCount number of months to add to given [SimDate]
	 * @return resulting [SimDate]
	 */
	fun addMonths(monthCount: Int) = of(date + monthCount)

	/**
	 * Add number of years to given [SimDate], return corresponding
	 * [SimDate]
	 *
	 * @param yearCount number of years to add to given [SimDate]
	 * @return resulting [SimDate]
	 */
	fun addYears(yearCount: Int) = of(date + yearCount * MONTH_PER_YEAR)

	/**
	 * Get the first month of the year for this [SimDate]
	 *
	 * @return first month of the year for this [SimDate]
	 */
	fun startOfYear() = of(this.year(), 1)

	/**
	 * Get the last month of the year for this [SimDate]
	 *
	 * @return last month of the year for this [SimDate]
	 */
	fun endOfYear() = of(this.year(), 12)

	operator fun plus(monthCount: Int) = addMonths(monthCount)

	operator fun minus(monthCount: Int) = addMonths(-monthCount)

	operator fun minus(other: SimDate) = other.monthsBetween(this)

	operator fun rangeTo(other: SimDate) = getSimDateList(this, other)

	/**
	 * How many months are between this [SimDate] and other [SimDate]
	 *
	 * @param other second [SimDate]
	 * @return months between this and other, i.e. (other - this) basically
	 */
	fun monthsBetween(other: SimDate): Int {
		return other.date - date
	}

	override fun compareTo(other: SimDate) = date - other.date

	fun isBefore(other: SimDate) = date < other.date

	fun isAfter(other: SimDate) = date > other.date

	override fun toString() = String.format("%04d.%02d", this.year(), this.month())

	companion object {

		const val MONTH_PER_YEAR = 12

		private val simDates: MutableMap<Int, SimDate> = mutableMapOf()

		/**
		 * Get the [SimDate] singleton for current month
		 *
		 * @return SimDate singleton
		 */
		@JvmStatic
		fun now(): SimDate {
			val now = LocalDate.now()
			return of(now.year, now.monthValue)
		}

		/**
		 * Get the [SimDate] singleton for given year and month
		 *
		 * @param year  year
		 * @param month month (1 .. 12)
		 * @return SimDate singleton
		 */
		@JvmStatic
		fun of(year: Int, month: Int): SimDate {
			require(month in 1..12) { "valid month" }
			return of(year * MONTH_PER_YEAR + month - 1)
		}

		/**
		 * Get the [SimDate] singleton for given localDate
		 *
		 * @param localDate localDate
		 * @return SimDate singleton
		 */
		@JvmStatic
		fun of(localDate: LocalDate) = of(localDate.year, localDate.monthValue)

		/**
		 * Get the [SimDate] singleton for given date string in format "yyyy.MM"
		 *
		 * @param date date string in format "yyyy.MM"
		 * @return SimDate singleton
		 */
		@JvmStatic
		fun of(date: String): SimDate {
			// check date format "yyyy.MM" with regex
			require(date.matches("\\d{4}\\.\\d{2}".toRegex())) { "valid date format" }
			val parts = date.split("\\.".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
			require(2 == parts.size) { "valid date" }
			val year = parts[0].toInt()
			val month = parts[1].toInt()
			return of(year, month)
		}

		/**
		 * Get the [SimDate] singleton for given simulation date
		 *
		 * @param date simulation date
		 * @return SimDate singleton
		 */
		private fun of(date: Int): SimDate {
			return simDates.computeIfAbsent(date) { d: Int -> SimDate(d) }
		}

		/**
		 * Return a dense list of monthly [SimDate]s between startDate and endDate
		 * [startDate .. endDate]
		 *
		 * @param startDate startDate (included)
		 * @param endDate   endDate (included)
		 * @return dense list of monthly [SimDate]s [startDate .. endDate]
		 */
		@JvmStatic
		fun getSimDateList(startDate: SimDate, endDate: SimDate): List<SimDate> {
			require(startDate <= endDate) { "valid date range" }
			val dateList: MutableList<SimDate> = ArrayList()
			for (i in startDate.date..endDate.date) {
				dateList.add(of(i))
			}
			return dateList
		}

		@JvmStatic
		fun min(a: SimDate, b: SimDate) = if (a < b) a else b

		@JvmStatic
		fun max(a: SimDate, b: SimDate) = if (a > b) a else b

	}

}
