package io.fincast.household.model.impl

import io.fincast.pos.model.ValueProviders.constValue
import io.fincast.pos.model.ValueProviders.eoyBalance
import io.fincast.household.model.Valuable
import io.fincast.household.model.enums.ProductType
import io.fincast.household.model.enums.RevenueAllocation
import io.fincast.pos.model.*
import io.fincast.pos.model.enums.Periodicity
import io.fincast.pos.model.impl.*

data class InvestmentAccount(
	override val tag: String,
	override val reconBalance: Double,

	val contributionAmount: Double? = null,
	val contributionPeriodicity: Periodicity? = null,
	val contributionStartDate: SimDate? = null,
	val contributionEndDate: SimDate? = null,

	val cashPart: Double = 0.0,
	val bondPart: Double = 0.0,
	val stockPart: Double = 0.0,
	val cashInterestRate: Double? = null,
	val bondInterestRate: Double? = null,
	val stockDividendYield: Double? = null,
	val stockCapitalGain: Double? = null,
	val revenueAllocation: RevenueAllocation = RevenueAllocation.PAYOUT,

	val managementFee: Double? = 0.0,
	val performanceFee: Double? = 0.0,

	val withdrawalAmount: Double? = null,
	val withdrawalPeriodicity: Periodicity? = null,
	val withdrawalStartDate: SimDate? = null,
	val withdrawalEndDate: SimDate? = null,
) : Valuable {

	override val productType = ProductType.ASSET

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		val compos: MutableList<PositionCompo> = mutableListOf()
		if (contributionAmount != null && contributionAmount > 0.0) {
			val startDate = SimDate.max(contributionStartDate ?: portfolio.reconDate, portfolio.reconDate)
			compos.add(
				TransferCompo(
					position = pos,
					tag = "contribution",
					fromPosition = portfolio.pocketMoney,
					toPosition = pos,
					amount = constValue(contributionAmount),
					sign = -1,
					startDate = startDate,
					endDate = contributionEndDate,
					periodicity = contributionPeriodicity ?: Periodicity.YEARLY,
				)
			)
		}
		if (cashPart != 0.0) {
			if (cashInterestRate != null && cashInterestRate != 0.0) {
				compos.add(
					InterestCompo(
						position = pos,
						tag = "cashInterest",
						revenueAllocation = revenueAllocation,
						amount = eoyBalance(pos),
						interestRate = constValue(cashPart / 100.0 * cashInterestRate)
					)
				)
			}
		}
		if (bondPart != 0.0) {
			if (bondInterestRate != null && bondInterestRate != 0.0) {
				compos.add(
					InterestCompo(
						position = pos,
						tag = "bondInterest",
						revenueAllocation = revenueAllocation,
						amount = eoyBalance(pos),
						interestRate = constValue(bondPart / 100.0 * bondInterestRate)
					)
				)
			}
		}
		if (stockPart != 0.0) {
			if (stockDividendYield != null && stockDividendYield != 0.0) {
				compos.add(
					DividendCompo(
						position = pos,
						tag = "dividend",
						revenueAllocation = revenueAllocation,
						dividendYield = constValue(stockPart / 100.0 * stockDividendYield)
					)
				)
			}
			if (stockCapitalGain != null && stockCapitalGain != 0.0) {
				compos.add(
					CapitalGainCompo(
						position = pos,
						tag = "capitalGain",
						capitalGainRate = constValue(stockPart / 100.0 * stockCapitalGain)
					)
				)
			}
		}
		if (managementFee != null && managementFee != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "managementFee",
					isFromPocket = false,
					amount = { managementFee / 100 * pos.getBalance(it) },
					sign = -1,
					periodicity = Periodicity.YEARLY,
				)
			)
		}
		if (performanceFee != null && performanceFee != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "performanceFee",
					isFromPocket = false,
					amount = { performanceFee / 100 * (pos.getBalance(it) - pos.getBalance(it.addYears(-1))) },
					sign = -1,
					periodicity = Periodicity.YEARLY,
				)
			)
		}
		if (withdrawalAmount != null && withdrawalAmount > 0.0) {
			val startDate = SimDate.max(withdrawalStartDate ?: portfolio.reconDate, portfolio.reconDate)
			compos.add(
				TransferCompo(
					position = pos,
					tag = "withdrawal",
					fromPosition = pos,
					toPosition = portfolio.pocketMoney,
					amount = constValue(withdrawalAmount),
					sign = -1,
					startDate = startDate,
					endDate = withdrawalEndDate,
					periodicity = withdrawalPeriodicity ?: Periodicity.YEARLY,
				)
			)
		}
		return compos
	}

}
