package io.dddrive.core.property.model.impl;

import static io.dddrive.util.Invariant.requireThis;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import io.dddrive.core.ddd.model.PartSPI;
import io.dddrive.core.enums.model.Enumerated;
import io.dddrive.core.enums.model.Enumeration;
import io.dddrive.core.property.model.EntityWithProperties;
import io.dddrive.core.property.model.EnumSetProperty;
import io.dddrive.core.property.model.base.PropertyBase;

public class EnumSetPropertyImpl<E extends Enumerated> extends PropertyBase<E> implements EnumSetProperty<E> {

	private final Enumeration<E> enumeration;

	private final Set<E> itemSet = new HashSet<>();

	public EnumSetPropertyImpl(EntityWithProperties entity, String name, Enumeration<E> enumeration) {
		super(entity, name);
		this.enumeration = enumeration;
	}

	@Override
	public void clearItems() {
		requireThis(this.isWritable(), "not frozen");
		this.itemSet.forEach(item -> ((PartSPI<?>) item).delete());
		this.itemSet.clear();
		this.getEntity().doAfterClear(this);
	}

	@Override
	public void addItem(E item) {
		requireThis(this.isWritable(), "not frozen");
		requireThis(item != null, "item not null");
		if (!this.hasItem(item)) {
			this.itemSet.add(item);
			this.getEntity().doAfterAdd(this, null);
		}
	}

	@Override
	public Set<E> getItems() {
		return Set.copyOf(this.itemSet);
	}

	@Override
	public boolean hasItem(E item) {
		return this.itemSet.contains(item);
	}

	@Override
	public void removeItem(E item) {
		requireThis(this.isWritable(), "not frozen");
		requireThis(item != null, "item not null");
		if (this.hasItem(item)) {
			this.itemSet.remove(item);
			this.getEntity().doAfterRemove(this);
		}
	}

	@Override
	public void loadItems(Collection<E> items) {
		this.itemSet.clear();
		this.itemSet.addAll(items);
	}

}
