package io.dddrive.core.property.model.base;

import static io.dddrive.util.Invariant.assertThis;
import static io.dddrive.util.Invariant.requireThis;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.dddrive.core.ddd.model.Aggregate;
import io.dddrive.core.ddd.model.AggregateRepository;
import io.dddrive.core.ddd.model.Part;
import io.dddrive.core.enums.model.Enumerated;
import io.dddrive.core.enums.model.Enumeration;
import io.dddrive.core.property.model.BaseProperty;
import io.dddrive.core.property.model.EntityWithProperties;
import io.dddrive.core.property.model.EnumProperty;
import io.dddrive.core.property.model.EnumSetProperty;
import io.dddrive.core.property.model.PartListProperty;
import io.dddrive.core.property.model.Property;
import io.dddrive.core.property.model.ReferenceProperty;
import io.dddrive.core.property.model.ReferenceSetProperty;
import io.dddrive.core.property.model.impl.BasePropertyImpl;
import io.dddrive.core.property.model.impl.EnumPropertyImpl;
import io.dddrive.core.property.model.impl.EnumSetPropertyImpl;
import io.dddrive.core.property.model.impl.PartListPropertyImpl;
import io.dddrive.core.property.model.impl.ReferencePropertyImpl;
import io.dddrive.core.property.model.impl.ReferenceSetPropertyImpl;

public abstract class EntityWithPropertiesBase implements EntityWithProperties {

	private final Map<String, Property<?>> propertyMap = new HashMap<>();

	@Override
	public boolean hasProperty(String name) {
		return this.propertyMap.containsKey(name);
	}

	@Override
	public Property<?> getProperty(String name) {
		return this.propertyMap.get(name);
	}

	@Override
	public List<Property<?>> getProperties() {
		return this.propertyMap.values().stream().toList();
	}

	protected void addProperty(Property<?> property) {
		requireThis(property.getName() != null, "property has name");
		requireThis(!this.hasProperty(property.getName()), "property [" + property.getName() + "] is unique");
		this.propertyMap.put(property.getName(), property);
	}

	protected <T> BaseProperty<T> addBaseProperty(String name, Class<T> type) {
		BaseProperty<T> property = new BasePropertyImpl<>(this, name);
		this.addProperty(property);
		return property;
	}

	protected <E extends Enumerated> EnumProperty<E> addEnumProperty(String name, Class<E> enumType) {
		Enumeration<E> enumeration = this.getDirectory().getEnumeration(enumType);
		assertThis(enumeration != null, "enumeration for " + enumType.getSimpleName() + " not null");
		EnumProperty<E> property = new EnumPropertyImpl<>(this, name, enumeration);
		this.addProperty(property);
		return property;
	}

	protected <E extends Enumerated> EnumSetProperty<E> addEnumSetProperty(String name, Class<E> enumType) {
		Enumeration<E> enumeration = this.getDirectory().getEnumeration(enumType);
		assertThis(enumeration != null, "enumeration for " + enumType.getSimpleName() + " not null");
		EnumSetProperty<E> property = new EnumSetPropertyImpl<>(this, name, enumeration);
		this.addProperty(property);
		return property;
	}

	protected <A extends Aggregate> ReferenceProperty<A> addReferenceProperty(String name, Class<A> aggregateType) {
		AggregateRepository<A> repo = this.getDirectory().getRepository(aggregateType);
		assertThis(repo != null, () -> "repo for " + aggregateType.getSimpleName() + " not null");
		ReferenceProperty<A> property = new ReferencePropertyImpl<>(this, name, repo::get);
		this.addProperty(property);
		return property;
	}

	protected <A extends Aggregate> ReferenceSetProperty<A> addReferenceSetProperty(String name, Class<A> aggregateType) {
		AggregateRepository<A> repo = this.getDirectory().getRepository(aggregateType);
		assertThis(repo != null, () -> "repo for " + aggregateType.getSimpleName() + " not null");
		ReferenceSetProperty<A> property = new ReferenceSetPropertyImpl<>(this, name, repo::get);
		this.addProperty(property);
		return property;
	}

	protected <P extends Part<?>> PartListProperty<P> addPartListProperty(String name, Class<P> partType) {
		PartListProperty<P> property = new PartListPropertyImpl<>(this, name, partType);
		this.addProperty(property);
		return property;
	}

}
