package io.dddrive.core.ddd.model.base;

import static io.dddrive.util.Invariant.assertThis;

import io.dddrive.core.ddd.model.Aggregate;
import io.dddrive.core.ddd.model.Part;
import io.dddrive.core.ddd.model.PartMeta;
import io.dddrive.core.ddd.model.PartSPI;
import io.dddrive.core.ddd.model.RepositoryDirectory;
import io.dddrive.core.property.model.BaseProperty;
import io.dddrive.core.property.model.EntityWithProperties;
import io.dddrive.core.property.model.Property;
import io.dddrive.core.property.model.base.EntityWithPropertiesBase;

public abstract class PartBase<A extends Aggregate>
		extends EntityWithPropertiesBase
		implements Part<A>, PartMeta<A>, PartSPI<A> {

	protected final BaseProperty<Integer> id = this.addBaseProperty("id", Integer.class);

	private final A aggregate;
	private boolean isChanged = false;
	private int isCalcDisabled = 0;
	private boolean isInCalc = false;
	private boolean didCalcAll = false;
	private boolean didCalcVolatile = false;

	protected PartBase(A aggregate, Integer id) {
		this.aggregate = aggregate;
		this.id.setValue(id);
	}

	@Override
	public RepositoryDirectory getDirectory() {
		return this.getAggregate().getMeta().getRepository().getDirectory();
	}

	@Override
	public A getAggregate() {
		return this.aggregate;
	}

	@Override
	public Integer getId() {
		return this.id.getValue();
	}

	@Override
	public boolean isFrozen() {
		return ((EntityWithProperties) this.getAggregate()).isFrozen();
	}

	@Override
	public Part<?> doAddPart(Property<?> property) {
		assertThis(false, "could instantiate part for property " + this.getClassName() + "." + property.getName());
		return null;
	}

	@Override
	public void doAfterSet(Property<?> property) {
		this.calcAll();
	}

	@Override
	public void doAfterAdd(Property<?> property, Part<?> part) {
		this.calcAll();
	}

	@Override
	public void doAfterRemove(Property<?> property) {
		this.calcAll();
	}

	@Override
	public void doAfterClear(Property<?> property) {
		this.calcAll();
	}

	@Override
	public boolean isChanged() {
		return this.isChanged;
	}

	@Override
	public boolean isCalcEnabled() {
		return this.isCalcDisabled == 0 && this.getAggregate().getMeta().isCalcEnabled();
	}

	@Override
	public void disableCalc() {
		this.isCalcDisabled += 1;
	}

	@Override
	public void enableCalc() {
		this.isCalcDisabled -= 1;
	}

	protected Boolean isInCalc() {
		return this.isInCalc;
	}

	protected void beginCalc() {
		this.isInCalc = true;
		this.didCalcAll = false;
		this.didCalcVolatile = false;
	}

	protected void endCalc() {
		this.isInCalc = false;
	}

	@Override
	public void calcAll() {
		this.isChanged = true;
		if (!this.isCalcEnabled() || this.isInCalc()) {
			return;
		}
		try {
			this.beginCalc();
			this.doCalcAll();
			this.getAggregate().calcAll();
			assertThis(this.didCalcAll, this.getClassName() + ": doCalcAll was propagated");
		} finally {
			this.endCalc();
		}
	}

	protected void doCalcAll() {
		this.didCalcAll = true;
	}

	@Override
	public void calcVolatile() {
		if (!this.isCalcEnabled() || this.isInCalc()) {
			return;
		}
		try {
			this.beginCalc();
			this.doCalcVolatile();
			assertThis(this.didCalcVolatile, this.getClassName() + ": doCalcAll was propagated");
		} finally {
			this.endCalc();
		}
	}

	protected void doCalcVolatile() {
		this.didCalcVolatile = true;
	}

	private String getClassName() {
		return this.getClass().getSuperclass().getSimpleName();
	}

}
