package io.dddrive.ddd.service.api.base;

import java.util.Map;

import org.springframework.context.event.EventListener;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.stats.CacheStats;

import io.dddrive.app.event.AggregateStoredEvent;
import io.dddrive.app.service.api.AppContextSPI;
import io.dddrive.ddd.model.Aggregate;
import io.dddrive.ddd.model.AggregateRepository;
import io.dddrive.ddd.service.api.AggregateCache;
import io.dddrive.enums.adapter.api.jsonapi.dto.EnumeratedDto;

public abstract class AggregateCacheBase<A extends Aggregate> implements AggregateCache<A> {

	private final AggregateRepository<? extends A, ?> repository;
	private final Cache<Integer, A> objCache = Caffeine.newBuilder().maximumSize(100).recordStats().build();
	private final Cache<Integer, EnumeratedDto> enumCache = Caffeine.newBuilder().maximumSize(100).recordStats().build();

	private int objCacheClear = 0;
	private int enumCacheClear = 0;

	public AggregateCacheBase(AggregateRepository<? extends A, ?> repository, Class<A> aggregateClass) {
		this.repository = repository;
		((AppContextSPI) repository.getAppContext()).addCache(aggregateClass, this);
	}

	protected AggregateRepository<? extends A, ?> getRepository() {
		return this.repository;
	}

	@Override
	public A get(Integer id) {
		if (id == null) {
			return null;
		}
		A aggregate = this.objCache.get(id, (aggrId) -> this.repository.get(aggrId));
		return aggregate;
	}

	@Override
	public A get(String key) {
		if (key == null) {
			return null;
		}
		Integer aggrId = this.repository.getId(key);
		return this.get(aggrId);
	}

	public Map<String, Integer> getStatistics() {
		CacheStats objStats = this.objCache.stats();
		CacheStats enumStats = this.enumCache.stats();
		return Map.of(
				"objCacheHit", (int) objStats.hitCount(),
				"objCacheMiss", (int) objStats.missCount(),
				"objCacheClear", this.objCacheClear,
				"enumCacheHit", (int) enumStats.hitCount(),
				"enumCacheMiss", (int) enumStats.missCount(),
				"enumCacheClear", this.enumCacheClear);
	}

	@EventListener
	public void handleAggregateStoredEvent(AggregateStoredEvent event) {
		Integer id = event.getAggregate().getId();
		if (this.objCache.getIfPresent(id) != null) {
			this.objCacheClear += 1;
			this.objCache.invalidate(id);
		}
		if (this.enumCache.getIfPresent(id) != null) {
			this.enumCacheClear += 1;
			this.enumCache.invalidate(id);
		}
	}

}
