
package io.dddrive.ddd.model.base;

import static io.dddrive.util.Invariant.assertThis;
import static io.dddrive.util.Invariant.requireThis;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import io.dddrive.ddd.model.Aggregate;
import io.dddrive.ddd.model.AggregateMeta;
import io.dddrive.ddd.model.AggregateRepository;
import io.dddrive.ddd.model.Part;
import io.dddrive.ddd.model.PartRepository;
import io.dddrive.ddd.model.enums.CodePartListType;
import io.dddrive.property.model.PartListProperty;
import io.dddrive.property.model.Property;
import io.dddrive.property.model.PropertyProvider;
import io.dddrive.property.model.base.EntityWithPropertiesBase;
import io.dddrive.validation.model.AggregatePartValidation;
import io.dddrive.validation.model.enums.CodeValidationLevel;
import io.dddrive.validation.model.impl.AggregatePartValidationImpl;

/**
 * A DDD Aggregate
 */
public abstract class AggregateBase extends EntityWithPropertiesBase implements Aggregate, AggregateMeta, AggregateSPI {

	private final AggregateRepository<? extends Aggregate, ? extends Object> repository;
	private final Object state;

	private Map<Class<? extends Part<?>>, PartCache<?>> partCaches = new ConcurrentHashMap<>();

	private boolean isFrozen = true;
	private int isCalcDisabled = 0;
	private boolean isInCalc = false;
	private List<AggregatePartValidation> validations = new ArrayList<>();

	private boolean didCalcAll = false;
	private boolean didCalcVolatile = false;

	protected Integer doInitSeqNr = 0;
	protected Integer doAfterCreateSeqNr = 0;
	protected Integer doAssignPartsSeqNr = 0;
	protected Integer doAfterLoadSeqNr = 0;
	protected Integer doBeforeStoreSeqNr = 0;
	protected Integer doCalcSearchSeqNr = 0;
	protected Integer doAfterStoreSeqNr = 0;

	private List<String> searchTexts = new ArrayList<>();
	private List<String> searchTokens = new ArrayList<>();

	protected AggregateBase(AggregateRepository<? extends Aggregate, ? extends Object> repository, Object state) {
		this.repository = repository;
		this.state = state;
	}

	@Override
	public AggregateRepository<? extends Aggregate, ? extends Object> getRepository() {
		return this.repository;
	}

	@Override
	public Object getAggregateState() {
		return this.state;
	}

	@Override
	public AggregateMeta getMeta() {
		return this;
	}

	@Override
	public final PropertyProvider getPropertyProvider() {
		if (this.getRepository() != null) { // possibly null in instatiation phase
			return ((AggregateRepositorySPI<?, ?>) this.getRepository()).getPropertyProvider();
		}
		return null;
	}

	public boolean hasPartCache(Class<? extends Part<?>> clazz) {
		return this.partCaches.containsKey(clazz);
	}

	public void initPartCache(Class<? extends Part<?>> clazz) {
		requireThis(!this.hasPartCache(clazz), "aggregate not yet initialised"
				+ " (did you forget to registerPartRepositories() in repository?)");
		this.partCaches.put(clazz, new PartCache<>());
	}

	public PartCache<?> getPartCache(Class<? extends Part<?>> clazz) {
		requireThis(this.hasPartCache(clazz), "part cache for " + clazz.getSimpleName() + " initialised"
				+ " (did you forget to registerPartRepositories() in repository?)");
		return this.partCaches.get(clazz);
	}

	@Override
	public void doInit(Integer aggregateId, Integer tenantId) {
		this.doInitSeqNr += 1;
	}

	@Override
	public void doAfterCreate() {
		this.doAfterCreateSeqNr += 1;
	}

	@Override
	public void doAssignParts() {
		this.doAssignPartsSeqNr += 1;
		for (Property<?> property : this.getProperties()) {
			if (property instanceof PartListProperty<?> partListProperty) {
				this.assignPartListParts(partListProperty, partListProperty.getPartListType());
			}
		}
	}

	@SuppressWarnings("unchecked")
	private <A extends Aggregate, P extends Part<A>> void assignPartListParts(PartListProperty<?> property,
			CodePartListType partListType) {
		Class<P> partType = ((PartListProperty<P>) property).getPartType();
		PartRepository<A, P> partRepository = this.getRepository().getPartRepository(partType);
		List<P> partList = partRepository.getParts((A) this, partListType);
		property.loadParts(partList);
	}

	@Override
	public void doAfterLoad() {
		this.doAfterLoadSeqNr += 1;
	}

	@Override
	public void doBeforeStore() {
		this.doBeforeStoreSeqNr += 1;
		this.doBeforeStoreProperties();
	}

	protected void doStoreSearch() {
		this.searchTexts.clear();
		this.searchTokens.clear();
		this.doCalcSearch();
		((AggregateRepositoryBase<?, ?>) this.getRepository()).storeSearch(this, this.searchTexts, this.searchTokens);
	}

	protected void addSearchText(String text) {
		this.searchTexts.add(text);
	}

	protected void addSearchTexts(List<String> texts) {
		this.searchTexts.addAll(texts);
	}

	protected void addSearchToken(String token) {
		this.searchTokens.add(token);
	}

	protected void addSearchTokens(List<String> tokens) {
		this.searchTokens.addAll(tokens);
	}

	@Override
	public void doCalcSearch() {
		this.doCalcSearchSeqNr += 1;
	}

	@Override
	public void doAfterStore() {
		this.doAfterStoreSeqNr += 1;
	}

	@Override
	public boolean isFrozen() {
		return this.isFrozen;
	}

	protected void unfreeze() {
		this.isFrozen = false;
	}

	protected void freeze() {
		this.isFrozen = true;
	}

	@Override
	public Part<?> doAddPart(Property<?> property, CodePartListType partListType) {
		if (property instanceof PartListProperty<?>) {
			return this.addPartListPart(property, partListType);
		}
		assertThis(false, "could instantiate part for property " + property.getName() + " (" + partListType.getId() + ", "
				+ property.getClass().getSimpleName() + ")");
		return null;
	}

	@SuppressWarnings("unchecked")
	private <A extends Aggregate, P extends Part<A>> Part<A> addPartListPart(Property<?> property,
			CodePartListType partListType) {
		Class<P> partType = ((PartListProperty<P>) property).getPartType();
		PartRepository<A, P> partRepository = this.getRepository().getPartRepository(partType);
		return partRepository.create((A) this, partListType);
	}

	@Override
	public void doAfterSet(Property<?> property) {
		this.calcAll();
	}

	@Override
	public void doAfterAdd(Property<?> property, Part<?> part) {
		this.calcAll();
	}

	@Override
	public void doAfterRemove(Property<?> property) {
		this.calcAll();
	}

	@Override
	public void doAfterClear(Property<?> property) {
		this.calcAll();
	}

	private void clearValidationList() {
		this.validations.clear();
	}

	@Override
	public List<AggregatePartValidation> getValidations() {
		return List.copyOf(this.validations);
	}

	protected void addValidation(CodeValidationLevel validationLevel, String validation) {
		this.validations.add(new AggregatePartValidationImpl(this.validations.size(), validationLevel, validation));
	}

	@Override
	public boolean isCalcEnabled() {
		return this.isCalcDisabled == 0;
	}

	@Override
	public void disableCalc() {
		this.isCalcDisabled += 1;
	}

	@Override
	public void enableCalc() {
		this.isCalcDisabled -= 1;
	}

	protected boolean isInCalc() {
		return this.isInCalc;
	}

	protected void beginCalc() {
		this.isInCalc = true;
		this.didCalcAll = false;
		this.didCalcVolatile = false;
	}

	protected void endCalc() {
		this.isInCalc = false;
	}

	@Override
	public void calcAll() {
		if (!this.isCalcEnabled() || this.isInCalc()) {
			return;
		}
		try {
			this.beginCalc();
			this.clearValidationList();
			this.doCalcAll();
			assertThis(this.didCalcAll, this.getClass().getSimpleName() + ": doCalcAll was propagated");
		} finally {
			this.endCalc();
		}
	}

	protected void doCalcAll() {
		this.didCalcAll = true;
	}

	@Override
	public void calcVolatile() {
		if (!this.isCalcEnabled() || this.isInCalc()) {
			return;
		}
		try {
			this.beginCalc();
			this.doCalcVolatile();
			assertThis(this.didCalcVolatile, this.getClass().getSimpleName() + ": doCalcAll was propagated");
		} finally {
			this.endCalc();
		}
	}

	protected void doCalcVolatile() {
		this.didCalcVolatile = true;
	}

}
