
package io.dddrive.ddd.model;

import java.util.List;

import io.crnk.core.queryspec.QuerySpec;
import io.dddrive.app.model.RequestContext;
import io.dddrive.app.service.api.AppContext;
import io.dddrive.ddd.model.enums.CodeAggregateType;
import io.dddrive.ddd.service.api.AggregateCache;
import io.dddrive.enums.model.Enumerated;
import io.dddrive.enums.model.Enumeration;

/**
 * A DDD Aggregate Repository
 */
public interface AggregateRepository<A extends Aggregate, V extends Object> {

	/**
	 * Get aggregate type
	 */
	CodeAggregateType getAggregateType();

	/**
	 * Get aggregate class
	 */
	Class<A> getAggregateClass();

	/**
	 * Get application context
	 */
	AppContext getAppContext();

	/**
	 * Get request context
	 */
	RequestContext getRequestContext();

	<Aggr extends Aggregate> AggregateRepository<Aggr, ?> getRepository(Class<Aggr> intfClass);

	<Aggr extends Aggregate> AggregateCache<Aggr> getCache(Class<Aggr> intfClass);

	<Aggr extends Aggregate, P extends Part<Aggr>> PartRepository<Aggr, P> getPartRepository(Class<P> intfClass);

	<E extends Enumerated> Enumeration<E> getEnumeration(Class<E> enumClass);

	/**
	 * Create a new Aggregate instance
	 *
	 * @param tenantId the tenant in which to create the instance (could be
	 *                 different from session, e.g. new Tenant in
	 *                 Kernel-Admin-Session)
	 */
	A create(Integer tenantId);

	/**
	 * Get a read-only Aggregate with given id
	 * 
	 * @return aggregate, it will be frozen
	 * @throws NoDataFound exception when aggregate not found
	 */
	A get(Integer id);

	/**
	 * Get Agregate ID by given key
	 * 
	 * @return aggregateId
	 * @throws NoDataFound exception when aggregate not found
	 */
	Integer getId(String key);

	/**
	 * Get a writeable Aggregate with given id
	 * 
	 * @return aggregate (not frozen)
	 * @throws NoDataFound exception when aggregate not found
	 */
	A load(Integer id);

	/**
	 * Store the Aggregate
	 */
	void store(A aggregate);

	/**
	 * Store the Aggregate search index
	 */
	void storeSearch(A aggregate);

	/**
	 * Find list of Aggregates matching search criteria
	 */
	List<V> find(QuerySpec querySpec);

	/**
	 * Get a list of Aggregates with the given foreign key pointing to targetId
	 */
	List<V> getByForeignKey(String fkName, Integer targetId);

}
