package io.dddrive.property.model.impl;

import static io.dddrive.util.Invariant.assertThis;
import static io.dddrive.util.Invariant.requireThis;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import io.dddrive.ddd.model.Aggregate;
import io.dddrive.ddd.model.base.PartSPI;
import io.dddrive.ddd.model.enums.CodePartListType;
import io.dddrive.property.model.AggregatePartItem;
import io.dddrive.property.model.AggregateResolver;
import io.dddrive.property.model.ReferenceSetProperty;
import io.dddrive.property.model.base.EntityWithPropertiesSPI;
import io.dddrive.property.model.base.PropertyBase;

public class ReferenceSetPropertyImpl<A extends Aggregate> extends PropertyBase<A> implements ReferenceSetProperty<A> {

	private final String name;
	private final CodePartListType partListType;
	private Set<AggregatePartItem<?>> itemSet = new HashSet<>();

	public ReferenceSetPropertyImpl(EntityWithPropertiesSPI entity, String name, CodePartListType partListType,
			AggregateResolver<A> repository) {
		super(entity);
		this.name = name;
		this.partListType = partListType;
	}

	@Override
	public String getName() {
		return this.name;
	}

	@Override
	public CodePartListType getPartListType() {
		return this.partListType;
	}

	@Override
	public void clearItems() {
		requireThis(this.isWritable(), "not frozen");
		this.itemSet.forEach(item -> ((PartSPI<?>) item).delete());
		this.itemSet.clear();
		this.getEntity().afterClear(this);
	}

	@Override
	public void addItem(Integer id) {
		requireThis(this.isWritable(), "not frozen");
		requireThis(id != null, "aggregateId not null");
		if (id == null) {
			return; // make compiler happy (potential null pointer)
		}
		if (!this.hasItem(id)) {
			assertThis(this.isValidAggregateId(id), "valid aggregate id [" + id + "]");
			AggregatePartItem<?> part = (AggregatePartItem<?>) this.getEntity().addPart(this, this.partListType);
			assertThis(part != null,
					"entity " + this.getEntity().getClass().getSimpleName() + "created a part for " + this.partListType.getId()
							+ " (make sure to compare property with .equals() in addPart)");
			if (part == null) {
				return; // make compiler happy (potential null pointer)
			}
			part.setItemId(id.toString());
			this.itemSet.add(part);
			this.getEntity().afterAdd(this);
		}
	}

	@Override
	public Set<Integer> getItems() {
		return Set.copyOf(this.itemSet.stream().map(item -> Integer.valueOf(item.getItemId())).toList());
	}

	@Override
	public boolean hasItem(Integer aggregateId) {
		for (AggregatePartItem<?> part : this.itemSet) {
			if (part.getItemId().equals(aggregateId.toString())) {
				return true;
			}
		}
		return false;
	}

	@Override
	public void removeItem(Integer aggregateId) {
		requireThis(this.isWritable(), "not frozen");
		requireThis(aggregateId != null, "aggregateId not null");
		if (aggregateId == null) {
			return; // make compiler happy (potential null pointer)
		}
		if (this.hasItem(aggregateId)) {
			AggregatePartItem<?> part = this.itemSet.stream().filter(p -> p.getItemId().equals(aggregateId.toString()))
					.findAny()
					.get();
			((PartSPI<?>) part).delete();
			this.itemSet.remove(part);
			this.getEntity().afterRemove(this);
		}
	}

	public void doBeforeStore() {
		int seqNr = 0;
		for (Iterator<AggregatePartItem<?>> it = this.itemSet.iterator(); it.hasNext();) {
			it.next().setSeqNr(seqNr++);
		}
	}

	@Override
	public void loadReferences(Collection<? extends AggregatePartItem<?>> partList) {
		this.itemSet.clear();
		partList.forEach(p -> this.itemSet.add(p));
	}

	// TODO too expensive?
	private boolean isValidAggregateId(Integer id) {
		return true;
	}

}
