/**
 * vertigo - simple java starter
 *
 * Copyright (C) 2013-2017, KleeGroup, direction.technique@kleegroup.com (http://www.kleegroup.com)
 * KleeGroup, Centre d'affaire la Boursidiere - BP 159 - 92357 Le Plessis Robinson Cedex - France
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.vertigo.dynamox.domain.formatter;

import java.text.ParsePosition;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.List;
import java.util.function.BiFunction;

import io.vertigo.app.Home;
import io.vertigo.core.locale.LocaleManager;
import io.vertigo.dynamo.domain.metamodel.DataType;
import io.vertigo.dynamo.domain.metamodel.Formatter;
import io.vertigo.dynamo.domain.metamodel.FormatterException;
import io.vertigo.lang.Assertion;
import io.vertigo.lang.JsonExclude;
import io.vertigo.util.ListBuilder;
import io.vertigo.util.StringUtil;

/**
 * Gestion des formattages de dates.
 * Args contient plusieurs arguments séparés par des points virgules ';'
 *
 * Le premier argument est obligatoire il représente le format d'affichage d'une date .
 * Les arguments suivants sont facultatifs ils représentent les autres formats de saisie autorisés.
 * Par défaut le premier format de saisie autorisé est le format d'affichage.
 * En effet, pour des raisons ergonomiques il est toujours préférable de pouvoir saisir ce qui est affiché.
 *
 * Exemple 1 d'argument : "dd/MM/yyyy "
 *  On affiche la date au format dd/MM/yyyy
 *  En saisie on autorise dd/MM/yyyy

 * Exemple 2 d'argument : "dd/MM/yyyy ; dd/MM/yy"
 *  On affiche la date au format dd/MM/yyyy
 *  En saisie on autorise dd/MM/yyyy et dd/MM/yy
 *
 * @author pchretien
 */
public final class FormatterDate implements Formatter {
	/**
	 * Format(s) étendu(s) de la date en saisie.
	 * Cette variable n'est créée qu'au besoin.
	 */
	@JsonExclude
	private final List<String> patterns;

	/**
	 * Constructor.
	 */
	public FormatterDate(final String args) {
		// Les arguments ne doivent pas être vides.
		assertArgs(args != null);
		//-----
		final ListBuilder<String> patternsBuilder = new ListBuilder<>();
		for (final String token : args.split(";")) {
			patternsBuilder.add(token.trim());
		}

		//Saisie des dates
		//Le format d'affichage est le premier format de saisie autorisé
		//Autres saisies autorisées (facultatifs)
		patterns = patternsBuilder.unmodifiable().build();
		assertArgs(!patterns.isEmpty());
	}

	private static void assertArgs(final boolean test) {
		Assertion.checkArgument(test, "Les arguments pour la construction de FormatterDate sont invalides :format affichage;{autres formats de saisie}");
	}

	/** {@inheritDoc} */
	@Override
	public String valueToString(final Object objValue, final DataType dataType) {
		Assertion.checkArgument(dataType.isAboutDate(), "this formatter only applies on date formats");
		//-----
		if (objValue == null) {
			return ""; //Affichage d'une date non renseignée;
		}
		switch (dataType) {
			case Date:
				return dateToString((Date) objValue, patterns.get(0));
			case LocalDate:
				return localDateToString((LocalDate) objValue, patterns.get(0));
			case ZonedDateTime:
				return zonedDateTimeToString((ZonedDateTime) objValue, patterns.get(0));
			default:
				throw new IllegalStateException();
		}
	}

	/** {@inheritDoc} */
	@Override
	public Object stringToValue(final String strValue, final DataType dataType) throws FormatterException {
		Assertion.checkArgument(dataType.isAboutDate(), "Formatter ne s'applique qu'aux dates");
		//-----
		if (StringUtil.isEmpty(strValue)) {
			return null;
		}
		final String sValue = strValue.trim();
		switch (dataType) {
			case Date:
				return applyStringToObject(sValue, FormatterDate::doStringToDate);
			case LocalDate:
				return applyStringToObject(sValue, FormatterDate::doStringToLocalDate);
			case ZonedDateTime:
				return applyStringToObject(sValue, FormatterDate::doStringToZonedDateTime);
			default:
				throw new IllegalStateException();
		}
	}

	/*
	 *  Cycles through patterns to try and parse given String into a Date | LocalDate | ZonedDateTime
	 */
	private <T> T applyStringToObject(final String dateString, final BiFunction<String, String, T> fun) throws FormatterException {
		//StringToDate renvoit null si elle n'a pas réussi à convertir la date
		T dateValue = null;
		for (int i = 0; i < patterns.size() && dateValue == null; i++) {
			try {
				dateValue = fun.apply(dateString, patterns.get(i));
			} catch (final Exception e) {
				dateValue = null;
			}
		}
		//A null dateValue means all conversions have failed
		if (dateValue == null) {
			throw new FormatterException(Resources.DYNAMOX_DATE_NOT_FORMATTED);
		}
		return dateValue;
	}

	/*
	 * Converts a String to a LocalDate according to a given pattern
	 */
	private static LocalDate doStringToLocalDate(final String dateString, final String pattern) {
		final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(pattern);
		return LocalDate.parse(dateString, dateTimeFormatter);
	}

	/*
	 * Converts a String to a ZonedlDateTime according to a given pattern
	 */
	private static ZonedDateTime doStringToZonedDateTime(final String dateString, final String pattern) {
		final DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(pattern).withZone(ZoneId.of("UTC"));
		return ZonedDateTime.parse(dateString, dateTimeFormatter);
	}

	/*
	 * Converts a String to a java.util.Date according to a given pattern
	 */
	private static Date doStringToDate(final String dateString, final String pattern) {
		Date dateValue;

		//Formateur de date on le crée à chaque fois car l'implémentation de DateFormat est non synchronisé !
		final java.text.SimpleDateFormat formatter = new java.text.SimpleDateFormat(pattern, getLocaleManager().getCurrentLocale());
		formatter.setLenient(false);

		final ParsePosition parsePosition = new ParsePosition(0);
		dateValue = formatter.parse(dateString, parsePosition);

		//si le parsing n'a pas consommé toute la chaine, on refuse la conversion
		if (parsePosition.getIndex() != dateString.length()) {
			throw new IllegalStateException("Error parsing " + dateString + " with pattern :" + pattern + "at position " + parsePosition.getIndex());
		}
		return dateValue;
	}

	private static String dateToString(final Date date, final String pattern) {
		final java.text.SimpleDateFormat formatter = new java.text.SimpleDateFormat(pattern, getLocaleManager().getCurrentLocale());
		formatter.setLenient(false);
		return formatter.format(date);
	}

	private static String localDateToString(final LocalDate localDate, final String pattern) {
		return DateTimeFormatter.ofPattern(pattern)
				.format(localDate);
	}

	private static String zonedDateTimeToString(final ZonedDateTime zonedDateTime, final String pattern) {
		return DateTimeFormatter.ofPattern(pattern)
				.format(zonedDateTime);
	}

	private static LocaleManager getLocaleManager() {
		return Home.getApp().getComponentSpace().resolve(LocaleManager.class);
	}
}
