package io.ultreia.java4all.validation.maven.plugin;

/*-
 * #%L
 * Validation :: Maven plugin
 * %%
 * Copyright (C) 2021 - 2024 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.lang.Objects2;
import io.ultreia.java4all.lang.Strings;
import io.ultreia.java4all.util.ImportManager;
import io.ultreia.java4all.validation.impl.NuitonScopeValidatorImpl;
import io.ultreia.java4all.validation.impl.ValidationMessagesCollector;
import io.ultreia.java4all.validation.impl.definition.FieldValidatorDefinition;
import io.ultreia.java4all.validation.impl.definition.FileValidatorDefinition;
import io.ultreia.java4all.validation.impl.definition.FileValidatorEntryDefinition;
import io.ultreia.java4all.validation.impl.definition.ProjectValidatorDefinition;
import io.ultreia.java4all.validation.impl.field.FieldValidator;
import io.ultreia.java4all.validation.impl.io.ProjectValidatorDefinitionBuilder;
import io.ultreia.java4all.validation.impl.io.ProjectValidatorFileDefinitionHelper;
import io.ultreia.java4all.validation.impl.io.ProjectValidatorMappingHelper;
import io.ultreia.java4all.validation.spi.field.generator.FieldValidationGenerator;
import io.ultreia.java4all.validation.spi.field.generator.FieldValidatorGenerators;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.nuiton.validator.NuitonValidationContext;

import javax.annotation.Generated;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.stream.Collectors;

/**
 * Generates validators as Java files.
 * <p>
 * Created at 01/02/2024.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 2.0.0
 */
@Mojo(name = "generate-validators", defaultPhase = LifecyclePhase.GENERATE_SOURCES)
public class GenerateValidatorsMojo extends AbstractValidationMojo {

    public static final String CONTENT =
            "package %1$s;\n\n" +
                    "%2$s" +
                    "\n@Generated(\"io.ultreia.java4all.validation.maven.plugin.GenerateValidatorsMojo\")\n" +
                    "public class %3$s extends NuitonScopeValidatorImpl<%4$s>{\n" +
                    "    public %3$s() {\n" +
                    "        super(%4$s.class , %5$s, Set.of(new String[]{\n" +
                    "%6$s\n" +
                    "        }), %7$s);\n" +
                    "    }\n" +
                    "\n" +
                    "    @Override\n" +
                    "    protected void validate(%4$s bean, NuitonValidationContext validationContext, ValidationMessagesCollector messagesCollector) {\n" +
                    "%8$s\n" +
                    "    }\n\n" +
                    "%9$s\n" +
                    "}";
    /**
     * Directory where validators files are located.
     */
    @Parameter(defaultValue = "${basedir}/src/main/resources", required = true)
    protected File resources;
    /**
     * Directory where the validators files is located.
     */
    @Parameter(defaultValue = "${basedir}/src/main/resources", required = true)
    protected File src;
    /**
     * Directory where generate java validator files.
     */
    @Parameter(defaultValue = "${basedir}/target/generated-sources/java", required = true)
    protected File target;
    /**
     * Validation context type.
     */
    @Parameter(required = true)
    protected String validationContextType;

    private Set<String> missingGenerators = new TreeSet<>();

    @Override
    public void execute() throws MojoExecutionException {
        try (URLClassLoader loader = new URLClassLoader(new URL[]{src.toURI().toURL(), resources.toURI().toURL()}, getClass().getClassLoader())) {

            FieldValidatorGenerators generators = new FieldValidatorGenerators(loader);

            Map<String, String> classesMapping = new TreeMap<>();
            Path rootPath = target.toPath();
            ProjectValidatorDefinition projectValidatorDefinition = new ProjectValidatorDefinitionBuilder(loader).build(new ProjectValidatorFileDefinitionHelper().toLocation(src.toPath()).toUri().toURL());
            getLog().info(String.format("%d validator(s) detected.", projectValidatorDefinition.getFilesByEntry().size()));
            getLog().info(String.format("Generate at %s", rootPath));
            for (Map.Entry<FileValidatorEntryDefinition, FileValidatorDefinition> entry : projectValidatorDefinition.getFilesByEntry().entrySet()) {
                FileValidatorEntryDefinition key = entry.getKey();
                String filePackageName = key.getBeanType().getPackageName();
                String fileSimpleName = toSimpleClassName(key);
                String fileName = filePackageName + "." + fileSimpleName;
                classesMapping.put(key.getKey(), fileName);
                FileValidatorDefinition fileValidatorDefinition = entry.getValue();
                Path fileLocation = toLocation(rootPath, key, fileSimpleName);
                String fileContent = toContent(generators, key, filePackageName, fileSimpleName, fileValidatorDefinition);
                getLog().info(String.format("Generate %s.%s at %s", filePackageName, fileSimpleName, fileLocation));
                if (fileContent != null) {
                    if (Files.notExists(fileLocation.getParent())) {
                        Files.createDirectories(fileLocation.getParent());
                    }
                    Files.write(fileLocation, fileContent.getBytes(StandardCharsets.UTF_8));
                }
            }
            getLog().info(String.format("%s validator(s) generated.", classesMapping.size()));
            ProjectValidatorMappingHelper mappingHelper = new ProjectValidatorMappingHelper();

            getLog().info(String.format("Generate classes mapping at %s", mappingHelper.toLocation(target.toPath())));
            mappingHelper.write(classesMapping, target.toPath());

            if (!missingGenerators.isEmpty()) {
                throw new MojoExecutionException(String.format("There is %d missing generator(s):\n * %s", missingGenerators.size(), String.join("\n * ", missingGenerators)));
            }
        } catch (IOException | NoSuchMethodException e) {
            throw new MojoExecutionException(e);
        }
    }

    private String toSimpleClassName(FileValidatorEntryDefinition key) {
        return String.format("%sValidatorOn%sFor%s", key.getBeanType().getSimpleName(), (key.getContext() == null ? "" : Arrays.stream(key.getContext().split("-")).map(Strings::capitalize).collect(Collectors.joining(""))), Strings.capitalize(key.getScope()));
    }

    public Path toLocation(Path rootPath, FileValidatorEntryDefinition key, String fileSimpleName) {
        Path packagePath = rootPath.resolve(key.getBeanType().getPackageName().replaceAll("\\.", "/"));
        return packagePath.resolve(fileSimpleName + ".java");
    }

    private String toContent(FieldValidatorGenerators generators, FileValidatorEntryDefinition key, String filePackageName, String fileSimpleName, FileValidatorDefinition fileValidatorDefinition) throws NoSuchMethodException {
        ImportManager importManager = new ImportManager(filePackageName);
        importManager.addImport(NuitonScopeValidatorImpl.class);
        importManager.addImport(Generated.class);
        importManager.addImport(Set.class);
        importManager.addImport(NuitonValidationContext.class);
        importManager.addImport(ValidationMessagesCollector.class);
        importManager.addImport(key.getBeanType());
        Map<String, String> methods = new LinkedHashMap<>();
        int validatorsCount = fileValidatorDefinition.getFields().values().stream().mapToInt(List::size).sum();
        int validatorsIndex = 0;
        for (Map.Entry<String, List<FieldValidatorDefinition>> entry : fileValidatorDefinition.getFields().entrySet()) {
            String fieldName = entry.getKey();
            List<FieldValidatorDefinition> validatorDefinitions = entry.getValue();
            List<String> statements = new LinkedList<>();
            for (FieldValidatorDefinition validatorDefinition : validatorDefinitions) {
                statements.add(String.format("        // %d - %s\n", validatorsIndex, validatorDefinition.getComment()));

                String supplier = generateFieldGeneratorInstance(generators,importManager, key, fieldName, validatorDefinition);

                statements.add(String.format("        validator(%d, %s).validate(bean, validationContext, messagesCollector);\n", validatorsIndex, supplier));
                validatorsIndex++;
            }
            methods.put("validate" + Strings.capitalize(fieldName), String.join("", statements));
        }

        String simpleName = key.getBeanType().getSimpleName();
        return String.format(CONTENT,
                             filePackageName,
                             importManager.getImportsSection("\n"),
                             fileSimpleName,
                             simpleName,
                             (key.getContext() == null ? "null" : "\"" + key.getContext() + "\""),
                             fileValidatorDefinition.getFields().keySet().stream().map(s -> "                \"" + s + "\"").collect(Collectors.joining(",\n")),
                             validatorsCount,
                             methods.keySet().stream().map(m -> String.format("        %s(bean, validationContext, messagesCollector);", m)).collect(Collectors.joining("\n")),
                             methods.entrySet().stream().map(e -> String.format("    protected void %1$s(%2$s bean, NuitonValidationContext validationContext, ValidationMessagesCollector messagesCollector) {\n" +
                                                                                        "%3$s" +
                                                                                        "    }\n", e.getKey(), simpleName, e.getValue())).collect(Collectors.joining("\n")));
    }

    protected String generateFieldGeneratorInstance(FieldValidatorGenerators generators, ImportManager importManager, FileValidatorEntryDefinition key, String fieldName, FieldValidatorDefinition validatorDefinition) throws NoSuchMethodException {
        Class<? extends FieldValidator<?, ?>> validator = validatorDefinition.getValidator();
        Class<?> beanType = key.getBeanType();
        Optional<FieldValidationGenerator> optionalGenerator = generators.getGenerator(validator);
        if (optionalGenerator.isEmpty()) {
            getLog().error("No generator for type " + validator.getName());
            String getterName = FieldValidationGenerator.guessGetterName(beanType, fieldName);
            missingGenerators.add(validator.getName());
            return String.format("/* FIXME %s */() -> new io.ultreia.java4all.validation.impl.field.RequiredFieldValidator<>(\"%s\", %s::%s, null)", validator.getName(), fieldName, beanType.getSimpleName(), getterName);
        }
        return optionalGenerator.get().generate(key, validatorDefinition, Objects2.forName(validationContextType), importManager);
    }


}
