package io.ultreia.java4all.validation.impl.field;

/*-
 * #%L
 * Validation :: Impl
 * %%
 * Copyright (C) 2021 - 2024 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.i18n.I18n;
import org.nuiton.validator.NuitonValidationContext;

import java.util.Objects;
import java.util.function.Function;

/**
 * Created on 26/01/2024.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 2.0.0
 */
public abstract class FieldValidatorSupport<O, F> implements FieldValidator<O, F> {

    private final String fieldName;
    private final Function<O, F> fieldFunction;
    private final String messageKey;
    private final MessageBuilder<O, ? super NuitonValidationContext, ? super FieldValidator<O, ?>> messageBuilder;

    public FieldValidatorSupport(String fieldName, Function<O, F> fieldFunction, String messageKey, MessageBuilder<O, ? super NuitonValidationContext, ? super FieldValidator<O, ?>> messageBuilder) {
        this.fieldName = Objects.requireNonNull(fieldName);
        this.fieldFunction = fieldFunction;
        this.messageKey = messageKey;
        this.messageBuilder = messageBuilder;
    }

    public FieldValidatorSupport(String fieldName, Function<O, F> fieldFunction) {
        this.fieldName = Objects.requireNonNull(fieldName);
        this.fieldFunction = fieldFunction;
        this.messageKey = null;
        this.messageBuilder = null;
    }

    @Override
    public String getFieldName() {
        return fieldName;
    }

    @Override
    public F getField(O object) {
        return Objects.requireNonNull(fieldFunction()).apply(object);
    }

    @Override
    public String getMessage(O object, NuitonValidationContext validationContext) {
        if (messageKey == null) {
            return null;
        }
        if (messageBuilder == null) {
            return I18n.l(validationContext.getLocale(), messageKey);
        }
        return messageBuilder.build(object, validationContext, this, messageKey);
    }

    protected Function<O, F> fieldFunction() {
        return fieldFunction;
    }

}
