package org.nuiton.jaxx.widgets.temperature.validators;

/*-
 * #%L
 * JAXX :: Widgets
 * %%
 * Copyright (C) 2008 - 2021 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.opensymphony.xwork2.validator.ValidationException;
import com.opensymphony.xwork2.validator.validators.FieldValidatorSupport;
import org.nuiton.jaxx.validator.ValidatorUtil;
import org.nuiton.jaxx.widgets.temperature.TemperatureEditorModel;
import org.nuiton.jaxx.widgets.temperature.TemperatureFormat;

import java.util.LinkedHashMap;
import java.util.Map;

import static io.ultreia.java4all.i18n.I18n.n;

/**
 * To validate that the temperature is in correct bound.
 * <p>
 * Created by tchemit on 22/09/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
@SuppressWarnings("unused")
public class TemperatureBoundFieldValidator extends FieldValidatorSupport {

    /**
     * Map of results by format.
     */
    private final Map<String, TemperatureResult> temperatureResults = new LinkedHashMap<>();
    /**
     * Default temperature format (min and max are based on this format).
     */
    private TemperatureFormat defaultTemperatureFormat;
    /**
     * Min temperature.
     */
    private Float min;
    /**
     * Max temperature.
     */
    private Float max;

    static class TemperatureResult {
        private final float min;
        private final float max;
        private final String format;

        TemperatureResult(float min, float max, String format) {
            this.min = min;
            this.max = max;
            this.format = format;
        }

        public float getMin() {
            return min;
        }

        public float getMax() {
            return max;
        }

        public String getFormat() {
            return format;
        }
    }

    public void setDefaultTemperatureFormat(String defaultTemperatureFormat) {
        this.defaultTemperatureFormat = TemperatureFormat.valueOf(defaultTemperatureFormat);
    }

    public void setMin(String min) {
        this.min = Float.valueOf(min);
    }

    public void setMax(String max) {
        this.max = Float.valueOf(max);
    }

    public Map<String, TemperatureResult> getTemperatureResults() {
        return temperatureResults;
    }

    @Override
    public void validate(Object object) throws ValidationException {

        if (min == null) {
            throw new ValidationException("No parameter 'min' filled");
        }
        if (max == null) {
            throw new ValidationException("No parameter 'max' filled");
        }
        if (min >= max) {
            throw new ValidationException(String.format("No parameter 'min' (%s) is greater than 'max' (%s)", min, max));
        }
        String fieldName = getFieldName();
        if (fieldName == null) {
            throw new ValidationException("No parameter 'fieldName' filled");
        }
        Float temperature = (Float) getFieldValue(fieldName, object);
        if (temperature == null) {
            return;
        }
        temperatureResults.clear();

        @SuppressWarnings("unchecked") Map<String, TemperatureEditorModel> map = (Map<String, TemperatureEditorModel>) getFieldValue("temperatureEditorModels", object);
        TemperatureFormat temperatureFormat = defaultTemperatureFormat;
        if (map != null) {

            TemperatureEditorModel temperatureEditorModel = map.get(fieldName);
            //FIXME une validation est lancé dans l'ui alors que l'on ne devrait pas et on a pas ce composant alors
            if (temperatureEditorModel != null) {
                temperatureFormat = temperatureEditorModel.getFormat();
            }
        }

        if (temperature < min || temperature > max) {
            float min;
            float max;
            if (temperatureFormat.equals(defaultTemperatureFormat)) {
                min = this.min;
                max = this.max;
            } else {
                min = defaultTemperatureFormat.convert(this.min, temperatureFormat);
                max = defaultTemperatureFormat.convert(this.max, temperatureFormat);
            }
            String fieldBoundName = fieldName + "Bound";
            temperatureResults.put(fieldBoundName, new TemperatureResult(min, max, temperatureFormat.getLabel()));
            if (getDefaultMessage().isEmpty()) {
                String defaultMessage = String.format("%1$s##${%2$s.min}##${%2$s.max}##${%2$s.format}",
                                                      n("jaxx.validation.temperature.temperature.bound"), "temperatureResults." + fieldBoundName);
                setDefaultMessage(defaultMessage);
            }
            ValidatorUtil.addFieldError(this, stack, getFieldName(), object);
        }

    }

    @Override
    public String getValidatorType() {
        return "temperatureBound";
    }

}

