package io.ultreia.java4all.jaxx.widgets.length.nautical;

/*-
 * #%L
 * JAXX :: Widgets
 * %%
 * Copyright (C) 2008 - 2022 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.opensymphony.xwork2.validator.ValidationException;
import com.opensymphony.xwork2.validator.validators.FieldValidatorSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.nuiton.jaxx.validator.ValidatorUtil;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;

import static io.ultreia.java4all.i18n.I18n.n;

/**
 * Created on 21/06/2022.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 3.0.15
 */
public class NauticalLengthBoundFieldValidator extends FieldValidatorSupport {
    private static final Logger log = LogManager.getLogger(NauticalLengthBoundFieldValidator.class);

    /**
     * Map of results by format.
     */
    private final Map<String, NauticalLengthResult> results = new LinkedHashMap<>();
    /**
     * Default nauticalLength format (min and max are based on this format).
     */
    private NauticalLengthFormat defaultNauticalLengthFormat;
    /**
     * Min nauticalLength.
     */
    private Float min;
    /**
     * Max nauticalLength.
     */
    private Float max;

    static class NauticalLengthResult {
        private final float min;
        private final float max;
        private final String format;

        NauticalLengthResult(float min, float max, String format) {
            this.min = min;
            this.max = max;
            this.format = format;
        }

        public float getMin() {
            return min;
        }

        public float getMax() {
            return max;
        }

        public String getFormat() {
            return format;
        }
    }

    public void setDefaultNauticalLengthFormat(String defaultNauticalLengthFormat) {
        try {
            this.defaultNauticalLengthFormat = NauticalLengthFormat.valueOf(Objects.requireNonNull(defaultNauticalLengthFormat).toUpperCase());
        } catch (Exception e) {
            log.error(String.format("Could not set 'defaultNauticalLengthFormat' parameter with value '%s'", defaultNauticalLengthFormat), e);
            throw e;
        }
    }

    public void setMin(String min) {
        this.min = Float.valueOf(min);
    }

    public void setMax(String max) {
        this.max = Float.valueOf(max);
    }

    public Map<String, NauticalLengthResult> getResults() {
        return results;
    }

    @Override
    public void validate(Object object) throws ValidationException {

        if (defaultNauticalLengthFormat == null) {
            throw new ValidationException("No parameter 'defaultNauticalLengthFormat' filled");
        }
        if (min == null) {
            throw new ValidationException("No parameter 'min' filled");
        }
        if (max == null) {
            throw new ValidationException("No parameter 'max' filled");
        }
        if (min >= max) {
            throw new ValidationException(String.format("No parameter 'min' (%s) is greater than 'max' (%s)", min, max));
        }
        String fieldName = getFieldName();
        if (fieldName == null) {
            throw new ValidationException("No parameter 'fieldName' filled");
        }
        Float nauticalLength = (Float) getFieldValue(fieldName, object);
        if (nauticalLength == null) {
            return;
        }
        results.clear();

        @SuppressWarnings("unchecked") Map<String, NauticalLengthEditorModel> map = (Map<String, NauticalLengthEditorModel>) getFieldValue("nauticalLengthEditorModels", object);
        NauticalLengthFormat nauticalLengthFormat = defaultNauticalLengthFormat;
        if (map != null) {

            NauticalLengthEditorModel nauticalLengthEditorModel = map.get(fieldName);
            //FIXME une validation est lancé dans l'ui alors que l'on ne devrait pas et on a pas ce composant alors
            if (nauticalLengthEditorModel != null) {
                nauticalLengthFormat = nauticalLengthEditorModel.getFormat();
            }
        }

        if (nauticalLength < min || nauticalLength > max) {
            float min;
            float max;
            if (nauticalLengthFormat.equals(defaultNauticalLengthFormat)) {
                min = this.min;
                max = this.max;
            } else {
                min = defaultNauticalLengthFormat.convert(nauticalLengthFormat, this.min);
                max = defaultNauticalLengthFormat.convert(nauticalLengthFormat, this.max);
            }
            String fieldBoundName = fieldName + "Bound";
            results.put(fieldBoundName, new NauticalLengthBoundFieldValidator.NauticalLengthResult(min, max, nauticalLengthFormat.getLabel()));
            if (getDefaultMessage().isEmpty()) {
                String defaultMessage = String.format("%1$s##${%2$s.min}##${%2$s.max}##${%2$s.format}",
                                                      n("jaxx.validation.nauticalLength.nauticalLength.bound"), "results." + fieldBoundName);
                setDefaultMessage(defaultMessage);
            }
            ValidatorUtil.addFieldError(this, stack, getFieldName(), object);
        }

    }

    @Override
    public String getValidatorType() {
        return "nauticalLengthBound";
    }

}


