package io.ultreia.java4all.jaxx.widgets.validation.validator;

/*-
 * #%L
 * JAXX :: Widgets - Validation
 * %%
 * Copyright (C) 2008 - 2024 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.i18n.I18n;
import io.ultreia.java4all.jaxx.widgets.validation.JaxxWidgetsValidationContext;
import io.ultreia.java4all.validation.core.api.NuitonValidationContext;
import io.ultreia.java4all.validation.core.impl.ValidationMessagesCollector;
import io.ultreia.java4all.validation.core.impl.field.FieldValidatorSupport;
import org.nuiton.jaxx.widgets.temperature.TemperatureEditorModel;
import org.nuiton.jaxx.widgets.temperature.TemperatureFormat;

import java.util.Map;
import java.util.function.Function;

/**
 * Created on 01/29/2024.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 3.1.0
 */
public class TemperatureBoundFieldValidator<O> extends FieldValidatorSupport<O, Float> {

//    /**
//     * Map of results by format.
//     */
//    private final Map<String, TemperatureResult> temperatureResults = new LinkedHashMap<>();
    /**
     * Default temperature format (min and max are based on this format).
     */
    private final TemperatureFormat defaultTemperatureFormat;
    /**
     * Min temperature.
     */
    private final Float min;
    /**
     * Max temperature.
     */
    private final Float max;

    public TemperatureBoundFieldValidator(String fieldName, Function<O, Float> fieldFunction, TemperatureFormat defaultTemperatureFormat, Float min, Float max) {
        super(fieldName, fieldFunction);
        this.defaultTemperatureFormat = defaultTemperatureFormat;
        this.min = min;
        this.max = max;
    }

//    public Map<String, TemperatureResult> getTemperatureResults() {
//        return temperatureResults;
//    }

    @Override
    public void validate(O object, NuitonValidationContext validationContext, ValidationMessagesCollector messagesCollector) {

        if (min == null) {
            throw new IllegalStateException("No parameter 'min' filled");
        }
        if (max == null) {
            throw new IllegalStateException("No parameter 'max' filled");
        }
        if (min >= max) {
            throw new IllegalStateException(String.format("No parameter 'min' (%s) is greater than 'max' (%s)", min, max));
        }
        String fieldName = getFieldName();
        if (fieldName == null) {
            throw new IllegalStateException("No parameter 'fieldName' filled");
        }
        Float temperature = getField(object);
        if (temperature == null) {
            return;
        }
//        temperatureResults.clear();

        Map<String, TemperatureEditorModel> map = ((JaxxWidgetsValidationContext) validationContext).getTemperatureEditorModels();
        TemperatureFormat temperatureFormat = defaultTemperatureFormat;
        if (map != null) {

            TemperatureEditorModel temperatureEditorModel = map.get(fieldName);
            //FIXME une validation est lancé dans l'ui alors que l'on ne devrait pas et on a pas ce composant alors
            if (temperatureEditorModel != null) {
                temperatureFormat = temperatureEditorModel.getFormat();
            }
        }

        if (temperature < min || temperature > max) {
            float min;
            float max;
            if (temperatureFormat.equals(defaultTemperatureFormat)) {
                min = this.min;
                max = this.max;
            } else {
                min = defaultTemperatureFormat.convert(this.min, temperatureFormat);
                max = defaultTemperatureFormat.convert(this.max, temperatureFormat);
            }
//            String fieldBoundName = fieldName + "Bound";
            TemperatureResult result = new TemperatureResult(min, max, temperatureFormat.getLabel());
//            temperatureResults.put(fieldBoundName, result);
            addMessage(validationContext, messagesCollector, I18n.n("jaxx.validation.temperature.temperature.bound"), result.getMin(), result.getMax(), result.getFormat());
        }
    }

    public static class TemperatureResult {
        private final float min;
        private final float max;
        private final String format;

        TemperatureResult(float min, float max, String format) {
            this.min = min;
            this.max = max;
            this.format = format;
        }

        public float getMin() {
            return min;
        }

        public float getMax() {
            return max;
        }

        public String getFormat() {
            return format;
        }
    }

}

