package io.ultreia.java4all.jaxx.widgets.validation.validator;

/*-
 * #%L
 * JAXX :: Widgets - Validation
 * %%
 * Copyright (C) 2008 - 2024 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.i18n.I18n;
import io.ultreia.java4all.jaxx.widgets.length.nautical.NauticalLengthEditorModel;
import io.ultreia.java4all.jaxx.widgets.length.nautical.NauticalLengthFormat;
import io.ultreia.java4all.jaxx.widgets.validation.JaxxWidgetsValidationContext;
import io.ultreia.java4all.validation.core.api.NuitonValidationContext;
import io.ultreia.java4all.validation.core.impl.ValidationMessagesCollector;
import io.ultreia.java4all.validation.core.impl.field.FieldValidatorSupport;

import java.util.Map;
import java.util.function.Function;

/**
 * Created on 01/29/2024.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 3.1.0
 */
public class NauticalLengthBoundFieldValidator<O> extends FieldValidatorSupport<O, Float> {
//    /**
//     * Map of results by format.
//     */
//    private final Map<String, NauticalLengthResult> results = new LinkedHashMap<>();
    /**
     * Default nauticalLength format (min and max are based on this format).
     */
    private final NauticalLengthFormat defaultNauticalLengthFormat;
    /**
     * Min nauticalLength.
     */
    private final Float min;
    /**
     * Max nauticalLength.
     */
    private final Float max;

    public NauticalLengthBoundFieldValidator(String fieldName, Function<O, Float> fieldFunction, NauticalLengthFormat defaultNauticalLengthFormat, Float min, Float max) {
        super(fieldName, fieldFunction);
        this.defaultNauticalLengthFormat = defaultNauticalLengthFormat;
        this.min = min;
        this.max = max;
    }

//    public Map<String, NauticalLengthResult> getResults() {
//        return results;
//    }

    @Override
    public void validate(O object, NuitonValidationContext validationContext, ValidationMessagesCollector messagesCollector) {

        if (defaultNauticalLengthFormat == null) {
            throw new IllegalStateException("No parameter 'defaultNauticalLengthFormat' filled");
        }
        if (min == null) {
            throw new IllegalStateException("No parameter 'min' filled");
        }
        if (max == null) {
            throw new IllegalStateException("No parameter 'max' filled");
        }
        if (min >= max) {
            throw new IllegalStateException(String.format("No parameter 'min' (%s) is greater than 'max' (%s)", min, max));
        }
        String fieldName = getFieldName();
        Float nauticalLength = getField(object);
        if (nauticalLength == null) {
            return;
        }
//        results.clear();

        Map<String, NauticalLengthEditorModel> map = ((JaxxWidgetsValidationContext) validationContext).getNauticalLengthEditorModels();
        NauticalLengthFormat nauticalLengthFormat = defaultNauticalLengthFormat;
        if (map != null) {

            NauticalLengthEditorModel nauticalLengthEditorModel = map.get(fieldName);
            //FIXME une validation est lancé dans l'ui alors que l'on ne devrait pas et on a pas ce composant alors
            if (nauticalLengthEditorModel != null) {
                nauticalLengthFormat = nauticalLengthEditorModel.getFormat();
            }
        }

        if (nauticalLength < min || nauticalLength > max) {
            float min;
            float max;
            if (nauticalLengthFormat.equals(defaultNauticalLengthFormat)) {
                min = this.min;
                max = this.max;
            } else {
                min = defaultNauticalLengthFormat.convert(nauticalLengthFormat, this.min);
                max = defaultNauticalLengthFormat.convert(nauticalLengthFormat, this.max);
            }
//            String fieldBoundName = fieldName + "Bound";
            NauticalLengthResult result = new NauticalLengthResult(min, max, nauticalLengthFormat.getLabel());
//            results.put(fieldBoundName, result);
            addMessage(validationContext, messagesCollector, I18n.n("jaxx.validation.nauticalLength.nauticalLength.bound"), result.getMin(), result.getMax(), result.getFormat());
        }

    }

    public static class NauticalLengthResult {
        private final float min;
        private final float max;
        private final String format;

        NauticalLengthResult(float min, float max, String format) {
            this.min = min;
            this.max = max;
            this.format = format;
        }

        public float getMin() {
            return min;
        }

        public float getMax() {
            return max;
        }

        public String getFormat() {
            return format;
        }
    }

}


