package io.ultreia.java4all.jaxx.widgets.validation.validator;

/*-
 * #%L
 * JAXX :: Widgets - Validation
 * %%
 * Copyright (C) 2008 - 2024 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.auto.service.AutoService;
import io.ultreia.java4all.jaxx.widgets.length.nautical.NauticalLengthFormat;
import io.ultreia.java4all.util.ImportManager;
import io.ultreia.java4all.validation.core.api.NuitonValidationContext;
import io.ultreia.java4all.validation.core.impl.definition.FieldValidatorDefinition;
import io.ultreia.java4all.validation.core.impl.definition.FileValidatorEntryDefinition;
import io.ultreia.java4all.validation.impl.spi.field.FieldValidationGenerator;
import io.ultreia.java4all.validation.impl.spi.field.FieldValidatorGeneratorSupport;
import org.nuiton.jaxx.widgets.temperature.TemperatureFormat;

import java.util.List;

import static io.ultreia.java4all.validation.impl.spi.field.FieldValidationGenerator.escapeFloat;
import static io.ultreia.java4all.validation.impl.spi.field.FieldValidationGenerator.escapeString;

/**
 * To generate java field validators.
 * <p>
 * Created at 04/02/2024.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 3.1.0
 */
public final class FieldValidatorGenerators {
    private FieldValidatorGenerators() {
    }

    @AutoService(FieldValidationGenerator.class)
    public static class CoordinateLatitudeDtoFieldValidatorGenerator extends FieldValidatorGeneratorSupport {
        public CoordinateLatitudeDtoFieldValidatorGenerator() {
            super(CoordinateLatitudeDtoFieldValidator.class, true, true, false, false);
        }

        @Override
        protected void generateExtraParameters(FileValidatorEntryDefinition key, FieldValidatorDefinition validatorDefinition, Class<? extends NuitonValidationContext> validationContextType, ImportManager importManager, List<String> result) throws NoSuchMethodException {
            super.generateExtraParameters(key, validatorDefinition, validationContextType, importManager, result);
            result.add(escapeString(validatorDefinition.getParameter("editorName")));
        }
    }

    @AutoService(FieldValidationGenerator.class)
    public static class CoordinateLongitudeDtoFieldValidatorGenerator extends FieldValidatorGeneratorSupport {
        public CoordinateLongitudeDtoFieldValidatorGenerator() {
            super(CoordinateLongitudeDtoFieldValidator.class, true, true, false, false);
        }

        @Override
        protected void generateExtraParameters(FileValidatorEntryDefinition key, FieldValidatorDefinition validatorDefinition, Class<? extends NuitonValidationContext> validationContextType, ImportManager importManager, List<String> result) throws NoSuchMethodException {
            super.generateExtraParameters(key, validatorDefinition, validationContextType, importManager, result);
            result.add(escapeString(validatorDefinition.getParameter("editorName")));
        }
    }

    @AutoService(FieldValidationGenerator.class)
    public static class NauticalLengthBoundFieldValidatorGenerator extends FieldValidatorGeneratorSupport {
        public NauticalLengthBoundFieldValidatorGenerator() {
            super(NauticalLengthBoundFieldValidator.class, true, true, false, false);
        }

        @Override
        protected void generateExtraParameters(FileValidatorEntryDefinition key, FieldValidatorDefinition validatorDefinition, Class<? extends NuitonValidationContext> validationContextType, ImportManager importManager, List<String> result) throws NoSuchMethodException {
            super.generateExtraParameters(key, validatorDefinition, validationContextType, importManager, result);
            result.add(String.format("%s.%s", importManager.addImport(NauticalLengthFormat.class), validatorDefinition.getParameter("defaultNauticalLengthFormat").toUpperCase()));
            result.add(escapeFloat(validatorDefinition.getParameter("min")));
            result.add(escapeFloat(validatorDefinition.getParameter("max")));
        }
    }

    @AutoService(FieldValidationGenerator.class)
    public static class TemperatureBoundFieldValidatorGenerator extends FieldValidatorGeneratorSupport {
        public TemperatureBoundFieldValidatorGenerator() {
            super(TemperatureBoundFieldValidator.class, true, true, false, false);
        }

        @Override
        protected void generateExtraParameters(FileValidatorEntryDefinition key, FieldValidatorDefinition validatorDefinition, Class<? extends NuitonValidationContext> validationContextType, ImportManager importManager, List<String> result) throws NoSuchMethodException {
            super.generateExtraParameters(key, validatorDefinition, validationContextType, importManager, result);
            result.add(String.format("%s.%s", importManager.addImport(TemperatureFormat.class), validatorDefinition.getParameter("defaultTemperatureFormat").toUpperCase()));
            result.add(escapeFloat(validatorDefinition.getParameter("min")));
            result.add(escapeFloat(validatorDefinition.getParameter("max")));
        }
    }
}
