package io.ultreia.java4all.jaxx.widgets.validation.validator;

/*-
 * #%L
 * JAXX :: Widgets - Validation
 * %%
 * Copyright (C) 2008 - 2024 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.i18n.I18n;
import io.ultreia.java4all.jaxx.widgets.validation.JaxxWidgetsValidationContext;
import io.ultreia.java4all.validation.core.api.NuitonValidationContext;
import io.ultreia.java4all.validation.core.impl.ValidationMessagesCollector;
import io.ultreia.java4all.validation.core.impl.field.FieldValidatorSupport;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.jaxx.widgets.gis.CoordinateFormat;
import org.nuiton.jaxx.widgets.gis.absolute.AbsoluteDdCoordinateEditor;
import org.nuiton.jaxx.widgets.gis.absolute.AbsoluteDdCoordinateEditorModel;
import org.nuiton.jaxx.widgets.gis.absolute.AbsoluteDmdCoordinateEditor;
import org.nuiton.jaxx.widgets.gis.absolute.AbsoluteDmdCoordinateEditorModel;
import org.nuiton.jaxx.widgets.gis.absolute.AbsoluteDmsCoordinateEditor;
import org.nuiton.jaxx.widgets.gis.absolute.AbsoluteDmsCoordinateEditorModel;
import org.nuiton.jaxx.widgets.gis.absolute.CoordinatesEditor;

import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

/**
 * Created on 01/29/2024.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 3.1.0
 */
public class CoordinateLongitudeDtoFieldValidator<O> extends FieldValidatorSupport<O, Float> {

    private final String editorName;

    public CoordinateLongitudeDtoFieldValidator(String fieldName, Function<O, Float> fieldFunction, String editorName) {
        super(fieldName, fieldFunction);
        this.editorName = editorName;
    }

    @Override
    public void validate(O object, NuitonValidationContext validationContext, ValidationMessagesCollector messagesCollector) {

        if (StringUtils.isEmpty(editorName)) {
            throw new IllegalStateException("editorName is not defined");
        }

        if (getField(object) == null) {
            return;
        }
        Map<String, CoordinatesEditor> map = ((JaxxWidgetsValidationContext) validationContext).getCoordinatesEditors();

        if (map == null) {
            //FIXME For service validation, need to inject this map
            return;
        }
        CoordinatesEditor coordinatesEditor = map.get(editorName);
        //FIXME une validation est lancé dans l'ui alors que l'on ne devrait pas et on a pas ce composant alors
        if (coordinatesEditor == null) {
            return;
        }
        Objects.requireNonNull(coordinatesEditor, "can't find editor named " + editorName);

        CoordinateFormat format = coordinatesEditor.getModel().getFormat();
        switch (format) {
            case dd:
                validate(coordinatesEditor.getLongitudeDd(), validationContext, messagesCollector);
                break;
            case dms:
                validate(coordinatesEditor.getLongitudeDms(), validationContext, messagesCollector);
                break;
            case dmd:
                validate(coordinatesEditor.getLongitudeDmd(), validationContext, messagesCollector);
                break;
        }
    }

    private void validate(AbsoluteDdCoordinateEditor editor, NuitonValidationContext validationContext, ValidationMessagesCollector messagesCollector) {
        AbsoluteDdCoordinateEditorModel model = editor.getModel();
        Integer degree = model.getDegree();
        if (degree != null) {
            if (degree > 180) {
                addMessage(validationContext, messagesCollector, I18n.n("jaxx.validation.coordinate.degree.longitude.outOfBound"));
            }
        }
    }

    private void validate(AbsoluteDmsCoordinateEditor editor, NuitonValidationContext validationContext, ValidationMessagesCollector messagesCollector) {
        AbsoluteDmsCoordinateEditorModel model = editor.getModel();
        Integer degree = model.getDegree();
        if (degree != null) {
            if (degree > 180) {
                addMessage(validationContext, messagesCollector, I18n.n("jaxx.validation.coordinate.degree.longitude.outOfBound"));
            }
        }
        Integer minute = model.getMinute();
        if (minute != null) {

            if (minute > 59) {
                addMessage(validationContext, messagesCollector, I18n.n("jaxx.validation.coordinate.minute.longitude.outOfBound"));
            }
        }
        Integer second = model.getSecond();
        if (second != null) {
            if (second > 59) {
                addMessage(validationContext, messagesCollector, I18n.n("jaxx.validation.coordinate.second.longitude.outOfBound"));
            }
        }
    }

    private void validate(AbsoluteDmdCoordinateEditor editor, NuitonValidationContext validationContext, ValidationMessagesCollector messagesCollector) {

        AbsoluteDmdCoordinateEditorModel model = editor.getModel();
        Integer degree = model.getDegree();
        if (degree != null) {
            if (degree > 180) {
                addMessage(validationContext, messagesCollector, I18n.n("jaxx.validation.coordinate.degree.longitude.outOfBound"));
            }
        }
        Integer minute = model.getMinute();
        if (minute != null) {

            if (minute > 59) {
                addMessage(validationContext, messagesCollector, I18n.n("jaxx.validation.coordinate.minute.longitude.outOfBound"));
            }
        }

    }
}
