/*
 * #%L
 * JAXX :: Widgets Select
 * %%
 * Copyright (C) 2008 - 2020 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jaxx.widgets.select;

import com.google.common.collect.ImmutableSet;
import io.ultreia.java4all.decoration.Decorator;
import io.ultreia.java4all.lang.Setters;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.nuiton.jaxx.runtime.JAXXUtil;
import org.nuiton.jaxx.runtime.spi.UIHandler;
import org.nuiton.jaxx.runtime.swing.JAXXButtonGroup;
import org.nuiton.jaxx.runtime.swing.SwingUtil;
import org.nuiton.jaxx.runtime.swing.model.JaxxFilterableComboBoxModel;
import org.nuiton.jaxx.runtime.swing.renderer.DecoratorListCellRenderer;
import org.nuiton.jaxx.widgets.BeanUIUtil;
import org.nuiton.jaxx.widgets.select.actions.BeanFilterableComboBoxShowPopupAction;

import javax.swing.AbstractButton;
import javax.swing.ComboBoxEditor;
import javax.swing.FocusManager;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.plaf.basic.BasicComboBoxUI;
import javax.swing.text.JTextComponent;
import java.awt.Component;
import java.awt.FocusTraversalPolicy;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Le handler d'un {@link BeanFilterableComboBox}.
 * <p>
 * Note: ce handler n'est pas stateless et n'est donc pas partageable entre
 * plusieurs ui.
 *
 * @param <O> le type des objet contenus dans le modèle du composant.
 * @author Kevin Morin - morin@codelutin.com
 * @see BeanFilterableComboBox
 * @since 2.5.12
 */
@SuppressWarnings("unused")
public class BeanFilterableComboBoxHandler<O> implements PropertyChangeListener, UIHandler<BeanFilterableComboBox<O>> {

    public static final Logger log = LogManager.getLogger(BeanFilterableComboBoxHandler.class);
    protected BeanFilterableComboBox<O> ui;
    private final DocumentListener editorDocumentListener = new DocumentListener() {
        public void insertUpdate(DocumentEvent e) {
            updateFilter();
        }

        public void removeUpdate(DocumentEvent e) {
            updateFilter();
        }

        public void changedUpdate(DocumentEvent e) {
            updateFilter();
        }

        void updateFilter() {
            JComboBox<O> comboBox = ui.getCombobox();
            JaxxFilterableComboBoxModel<O> model = (JaxxFilterableComboBoxModel<O>) comboBox.getModel();
            JTextComponent editorComponent = (JTextComponent) comboBox.getEditor().getEditorComponent();
            // hide the popup before setting the filter, otherwise the popup height does not fit
            boolean wasPopupVisible = comboBox.isShowing() && comboBox.isPopupVisible();
            if (wasPopupVisible) {
                comboBox.hidePopup();
            }
            String text = editorComponent.getText();
            if (ui.getSelectedItem() != null) {
                text = "";
            }
            if (log.isDebugEnabled()) {
                log.debug("updateFilter " + text);
            }
            model.setFilterText(text);
            if (wasPopupVisible) {
                comboBox.showPopup();
            }
        }

    };
    private final BeanUIUtil.PopupHandler<O> popupHandler = new BeanUIUtil.PopupHandler<>() {

        @Override
        public JPopupMenu getPopup() {
            return ui.getPopup();
        }

        @Override
        public JComponent getInvoker() {
            return ui.getDisplayDecorator();
        }
    };
    /**
     * the mutator method on the property of boxed bean in the ui
     */
    protected Method mutator;
    /**
     * the decorator of data
     */
    protected Decorator decorator;
    protected boolean init;
    private JList<O> jList;
    private int selectedIndex;
    private Object selectedValue;
    /**
     * Magical focus owner used to see if previous focus owner was this widget.
     */
    private Object oldFocusOwner;
    /**
     * Enabled when editor has acquired the focus so key event can be used.
     */
    private boolean enabled = false;
    /**
     * When focus coming from another widget, we receive a TAB key event but this widget has already focus,
     * so it immediately pass the focus...
     * In a such case, le'ts consume the first TAB key event, then the real user TAB key event will be used.
     */
    private boolean tabHit = false;
    private final KeyAdapter documentKeyListener = new KeyAdapter() {

        final ImmutableSet<Integer> codesToSkip = ImmutableSet.of(
                KeyEvent.VK_F1,
                KeyEvent.VK_F2,
                KeyEvent.VK_F3,
                KeyEvent.VK_F4,
                KeyEvent.VK_F5,
                KeyEvent.VK_F6,
                KeyEvent.VK_F7,
                KeyEvent.VK_F8,
                KeyEvent.VK_F9,
                KeyEvent.VK_F10,
                KeyEvent.VK_ESCAPE,
                KeyEvent.VK_TAB,
                KeyEvent.VK_ENTER
        );

        @Override
        public void keyPressed(KeyEvent e) {
            if (!enabled || !ui.isEnabled()) {
                e.consume();
                return;
            }
            if (e.isControlDown() || e.isShiftDown() || e.isAltDown() || e.isAltGraphDown()) {
                return;
            }
            JComboBox<O> combobox = ui.getCombobox();
            log.debug(ui.getName() + "--keyPressed: " + e.getKeyCode());
            if (!combobox.isPopupVisible() && KeyEvent.VK_ENTER == e.getKeyCode()) {
                combobox.showPopup();
                e.consume();
            }
        }

        @Override
        public void keyReleased(KeyEvent e) {
            if (!enabled) {
                e.consume();
                return;
            }
            if (e.isControlDown() || e.isShiftDown() || e.isAltDown() || e.isAltGraphDown()) {
                return;
            }
            log.debug("permanent focus owner: " + FocusManager.getCurrentManager().getPermanentFocusOwner());
            log.debug("focus owner: " + FocusManager.getCurrentManager().getFocusOwner());
            if (!Objects.equals(FocusManager.getCurrentManager().getPermanentFocusOwner(), ui.getCombobox().getEditor().getEditorComponent())) {
                e.consume();
                return;
            }
            log.error("keyReleased: " + e.getKeyCode());
            JComboBox<O> combobox = ui.getCombobox();
            if (combobox.isPopupVisible() && KeyEvent.VK_ESCAPE == e.getKeyCode()) {
                log.debug("ESC , hide popup");
                e.consume();
                combobox.hidePopup();
                return;
            }
            if (!combobox.isPopupVisible() && !codesToSkip.contains(e.getKeyCode())) {
                log.debug("Will show popup, keycode: " + e.getKeyCode());
                combobox.showPopup();
            }
            // if the typed text does not match the selected item,
            // set the selected item to null
            Object selectedItem = ui.getSelectedItem();
            JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
            String text = editorComponent.getText();
            log.debug("keycode: " + e.getKeyCode() + ", editorText: " + text);
            if (KeyEvent.VK_ENTER == e.getKeyCode()) {

                log.debug("*ENTER* key");
                if (ui.isEnterToSelectUniqueUniverse() && combobox.getItemCount() == 1) {
                    // auto-select the
                    log.error("Auto-select unique result with *ENTER* key");
                    combobox.setSelectedIndex(0);
                    e.consume();
                    // edition is done
                    combobox.hidePopup();
                    return;
                }
            }
            boolean tabToSelect = ui.isTabToSelect();
            if (tabToSelect && KeyEvent.VK_TAB == e.getKeyCode()) {
                log.debug("*TAB* key");
                if (!acceptTabKeyEvent()) {
                    log.debug("*TAB* key rejected!!!");
                    e.consume();
                    return;
                }
                if (combobox.getItemCount() > 0) {
                    if (selectedIndex != -1) {
                        log.debug("Auto-select with *TAB* key");
                        combobox.setSelectedIndex(selectedIndex);
                    } else if (combobox.getItemCount() == 1) {
                        // auto-select the
                        log.debug("Auto-select unique result with *TAB* key");
                        combobox.setSelectedIndex(0);
                    }
                    // edition is done
                    combobox.hidePopup();
                    e.consume();
                }
                FocusTraversalPolicy focusTraversalPolicy = ui.getFocusCycleRootAncestor().getFocusTraversalPolicy();
                Component focusComponent =
                        e.isShiftDown() ? focusTraversalPolicy.getComponentBefore(ui.getFocusCycleRootAncestor(), combobox.getEditor().getEditorComponent())
                                : focusTraversalPolicy.getComponentAfter(ui.getFocusCycleRootAncestor(), combobox.getEditor().getEditorComponent());
                if (focusComponent != null) {
                    log.error("Change focus from " + ui.getName());
                    SwingUtilities.invokeLater(focusComponent::requestFocusInWindow);
                }
                return;
            }
            final String selectedItemString;
            if (getBeanType().isInstance(selectedItem)) {
                selectedItemString = BeanFilterableComboBoxHandler.this.decorator.decorate(selectedItem);
            } else {
                selectedItemString = JAXXUtil.getStringValue(selectedItem);
            }
            if (selectedItem == null || !selectedItemString.equals(text)) {
                unselectItem();
            }
        }

    };

    /**
     * Initialise le handler de l'ui
     *
     * @param decorator le decorateur a utiliser
     * @param data      la liste des données a gérer
     */
    public void init(Decorator decorator, List<O> data) {

        if (init) {
            throw new IllegalStateException("can not init the handler twice");
        }
        init = true;

        if (decorator == null) {
            throw new NullPointerException("decorator can not be null (for type " + ui.getBeanType() + ")");
        }

        JAXXButtonGroup indexes = ui.getIndexes();

        this.decorator = decorator.clone();

        JComboBox<O> combobox = ui.getCombobox();

        try {
            Field listBox = BasicComboBoxUI.class.getDeclaredField("listBox");
            listBox.setAccessible(true);
            //noinspection unchecked
            jList = (JList<O>) listBox.get(combobox.getUI());
        } catch (Exception e) {
            log.error("Can't get jList...", e);
        }
        jList.addListSelectionListener(e -> {
            selectedIndex = jList.getSelectedIndex();
            selectedValue = jList.getSelectedValue();
            log.debug("Selected index: " + selectedIndex + " - " + selectedValue);
        });

        JAXXFilterableComboBoxEditor editor =
                new JAXXFilterableComboBoxEditor(ui.getCombobox().getEditor());
        combobox.setEditor(editor);

        editor.getEditorComponent().addMouseListener(new MouseAdapter() {

            @Override
            public void mouseClicked(MouseEvent e) {
                if (ui.isEnabled()) {
                    combobox.showPopup();
                }
            }

        });
        if (ui.isTabToSelect()) {
            editor.getEditorComponent().setFocusTraversalKeysEnabled(false);
        }

        editor.getEditorComponent().addKeyListener(documentKeyListener);
        FocusManager.getCurrentManager().addPropertyChangeListener("permanentFocusOwner", this::onFocusChanged);
        // init combobox renderer base on given decorator
        combobox.setRenderer(new DecoratorListCellRenderer<>(this.decorator));
        ((JaxxFilterableComboBoxModel<O>) combobox.getModel()).setDecorator(this.decorator);
        combobox.putClientProperty("JComboBox.isTableCellEditor", Boolean.TRUE);
        combobox.addItemListener(e -> {
            Object item = e.getItem();
            if (e.getStateChange() == ItemEvent.SELECTED) {
                log.debug("itemStateChanged selected " + item + " - " + (item != null ? item.getClass() : null));
                combobox.getEditor().getEditorComponent().setForeground(null);
                ui.setSelectedItem(item);

            } else {
                log.debug("itemStateChanged deselected " + item + " - " + (item != null ? item.getClass() : null));
                combobox.getEditor().getEditorComponent().setForeground(ui.getInvalidComboEditorTextColor());
            }
        });

        // build popup
        popupHandler.preparePopup(ui.getSelectedToolTipText(),
                                  ui.getNotSelectedToolTipText(),
                                  ui.getI18nPrefix(),
                                  ui.getPopupTitleText(),
                                  indexes,
                                  ui.getPopupSeparator(),
                                  ui.getPopupLabel(),
                                  ui.getSortUp(),
                                  ui.getSortDown(),
                                  this.decorator);

        setFilterable(false, ui.getFilterable());

        ui.addPropertyChangeListener(this);

        // set datas
        ui.setData(data);

        // select sort button
        indexes.setSelectedButton(ui.getIndex());
    }

    /**
     * Toggle the popup visible state.
     */
    public void togglePopup() {
        popupHandler.togglePopup();
    }

    /**
     * @return {@code true} if there is no data in comboBox, {@code false}
     * otherwise.
     */
    public boolean isEmpty() {
        return CollectionUtils.isEmpty(ui.getData());
    }

    /**
     * Add the given items into the comboBox.
     * <p>
     * <strong>Note:</strong> The item will be inserted at his correct following
     * the selected ordering.
     *
     * @param items items to add in comboBox.
     * @since 2.5.28
     */
    public void addItems(Iterable<O> items) {

        List<O> data = ui.getData();

        boolean wasEmpty = CollectionUtils.isEmpty(data);

        for (O item : items) {
            data.add(item);
        }

        updateUI(ui.getIndex(), ui.isReverseSort());

        fireEmpty(wasEmpty);
    }

    /**
     * Remove the given items from the comboBox model.
     * <p>
     * <strong>Note:</strong> If this item was selected, then selection will be
     * cleared.
     *
     * @param items items to remove from the comboBox model
     * @since 2.5.28
     */
    public void removeItems(Iterable<O> items) {

        List<O> data = ui.getData();

        boolean needUpdate = false;
        for (O item : items) {
            boolean remove = data.remove(item);

            if (remove) {

                // item was found in data

                Object selectedItem = ui.getSelectedItem();
                if (item == selectedItem) {

                    // item was selected item, reset selected item then
                    ui.setSelectedItem(null);
                }

                needUpdate = true;
            }
        }

        if (needUpdate) {

            updateUI(ui.getIndex(), ui.isReverseSort());
            fireEmpty(false);
        }

    }

    /**
     * Add the given item into the comboBox.
     * <p>
     * <strong>Note:</strong> The item will be inserted at his correct following
     * the selected ordering.
     *
     * @param item item to add in comboBox.
     * @since 2.5.9
     */
    public void addItem(O item) {

        addItems(Collections.singleton(item));
    }

    /**
     * Remove the given item from the comboBox model.
     * <p>
     * <strong>Note:</strong> If this item was selected, then selection will be
     * cleared.
     *
     * @param item the item to remove from the comboBox model
     * @since 2.5.9
     */
    public void removeItem(O item) {

        removeItems(Collections.singleton(item));
    }

    /**
     * Sort data of the model.
     */
    public void sortData() {

        // just update UI should do the math of this
        updateUI(ui.getIndex(), ui.isReverseSort());
    }

    /**
     * Reset the combo-box; says remove any selected item and filter text.
     */
    public void reset() {
        if (ui.getSelectedItem() != null) {
            ui.setSelectedItem(null);
        } else {
            JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
            editorComponent.setText("");
        }

        JComboBox<O> comboBox = ui.getCombobox();
        if (comboBox.isShowing()) {
            comboBox.hidePopup();
        }
    }

    /**
     * Focus combo only if autoFocus ui property is on.
     *
     * @since 2.8.5
     */
    public void focusCombo() {
        if (ui.isAutoFocus()) {
            ui.combobox.requestFocusInWindow();
        }
    }

    /**
     * Modifie l'état filterable de l'ui.
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    protected void setFilterable(Boolean oldValue, Boolean newValue) {
        oldValue = oldValue != null && oldValue;
        newValue = newValue != null && newValue;
        if (oldValue.equals(newValue)) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("filterable state : <" + oldValue + " to " + newValue + ">");
        }
        if (!newValue) {
            JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
            editorComponent.getDocument().removeDocumentListener(editorDocumentListener);
            ((JaxxFilterableComboBoxModel<O>) ui.getCombobox().getModel()).setFilterText(null);

        } else {
            JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
            editorComponent.getDocument().addDocumentListener(editorDocumentListener);
            editorDocumentListener.changedUpdate(null);
        }
    }

    /**
     * Modifie l'index du décorateur
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    protected void setIndex(Integer oldValue, Integer newValue) {
        if (newValue == null || newValue.equals(oldValue)) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("check state : <" + oldValue + " to " + newValue + ">");
        }
        AbstractButton button = ui.getIndexes().getButton(newValue);
        if (button != null) {
            button.setSelected(true);
        }
        updateUI(newValue, ui.isReverseSort());
    }

    /**
     * Modifie l'index du décorateur
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    protected void setSortOrder(Boolean oldValue, Boolean newValue) {

        if (newValue == null || newValue.equals(oldValue)) {
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("check state : <" + oldValue + " to " + newValue + ">");
        }

        updateUI(ui.getIndex(), newValue);
    }

    protected void updateUI(int index, boolean reversesort) {

        // change decorator context
        decorator.setIndex(index);

        // keep selected item
        Object previousSelectedItem = ui.getSelectedItem();

        // remove autocomplete
        if (previousSelectedItem != null) {
            ui.getCombobox().setSelectedItem(null);
            ui.selectedItem = null;
        }

        List<O> data = ui.getData();

        if (ui.isSortable() && CollectionUtils.isNotEmpty(data)) {
            try {
                // Sort data with the decorator jxpath tokens.
                decorator.sort(data, index, reversesort);
            } catch (Exception eee) {
                log.warn(eee.getMessage(), eee);
            }
        }

        // reload the model
        SwingUtil.fillComboBox(ui.getCombobox(), data, null);

        if (previousSelectedItem != null) {
            ui.setSelectedItem(previousSelectedItem);
        }
    }

    protected void unselectItem() {
        if (ui.selectedItem == null) {
            return;
        }

        ui.selectedItem = null;
        BeanUIUtil.invokeMethod(getMutator(),
                                ui.getBean(),
                                (O) null);
    }

    /**
     * Modifie la valeur sélectionnée dans la liste déroulante.
     *
     * @param oldValue l'ancienne valeur
     * @param newValue la nouvelle valeur
     */
    protected void setSelectedItem(O oldValue, O newValue) {
        if (oldValue == null && newValue == null) {
            return;
        }

        if (!getBeanType().isInstance(newValue)) {
            newValue = null;
        }

        JTextComponent editorComponent = (JTextComponent) ui.getCombobox().getEditor().getEditorComponent();
        editorComponent.setText("");

        if (log.isDebugEnabled()) {
            log.debug(ui.getProperty() + " on " + getBeanType() + " :: " + oldValue + " to " + newValue);
        }

        BeanUIUtil.invokeMethod(getMutator(),
                                ui.getBean(),
                                newValue);
    }

    public Decorator getDecorator() {
        return decorator;
    }

    /**
     * @return get the type of objects contained in the comboBox model.
     */
    @SuppressWarnings("unchecked")
    public Class<O> getBeanType() {
        Class<O> result = ui.getBeanType();
        if (result == null) {
            result = decorator == null ? null : (Class<O>) decorator.definition().type();
        }
        return result;
    }

    /**
     * @return le mutateur a utiliser pour modifier le bean associé.
     */
    protected Method getMutator() {
        if (mutator == null && ui.getBean() != null && ui.getProperty() != null) {
            mutator = Setters.getMutator(ui.getBean(), ui.getProperty());
        }
        return mutator;
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        String propertyName = evt.getPropertyName();

        if (BeanFilterableComboBox.PROPERTY_SELECTED_ITEM.equals(propertyName)) {
            //noinspection unchecked
            setSelectedItem((O) evt.getOldValue(), (O) evt.getNewValue());
            return;
        }

        if (BeanFilterableComboBox.PROPERTY_FILTERABLE.equals(propertyName)) {

            setFilterable((Boolean) evt.getOldValue(),
                          (Boolean) evt.getNewValue());
            return;
        }

        if (BeanFilterableComboBox.PROPERTY_INDEX.equals(propertyName)) {

            // decorator index has changed, force reload of data in ui
            setIndex((Integer) evt.getOldValue(),
                     (Integer) evt.getNewValue());
            return;
        }

        if (BeanFilterableComboBox.PROPERTY_REVERSE_SORT.equals(propertyName)) {

            // sort order has changed, force reload of data in ui
            setSortOrder((Boolean) evt.getOldValue(),
                         (Boolean) evt.getNewValue());
            return;
        }

        if (BeanFilterableComboBox.PROPERTY_DATA.equals(propertyName)) {

            // list has changed, force reload of index
            setIndex(null, ui.getIndex());

            // list has changed, fire empty property
            List<?> list = (List<?>) evt.getOldValue();
            fireEmpty(CollectionUtils.isEmpty(list));
        }
    }

    protected void fireEmpty(boolean wasEmpty) {
        ui.firePropertyChange(BeanComboBox.PROPERTY_EMPTY, wasEmpty,
                              isEmpty());
    }

    @Override
    public void beforeInit(BeanFilterableComboBox<O> ui) {
        this.ui = ui;
    }

    @Override
    public void afterInit(BeanFilterableComboBox<O> ui) {
        BeanFilterableComboBoxShowPopupAction.init(ui, null, new BeanFilterableComboBoxShowPopupAction<>());
    }

    //FIXME Remove all the logging and try to improve the hack...

    private boolean acceptTabKeyEvent() {
        if (!enabled) {
            log.debug("Reject TAB: not enabled: " + ui.getName());
            return false;
        }
        if (oldFocusOwner == null) {
            return true;
        }
        if (!(oldFocusOwner instanceof JTextField)) {
            if (tabHit) {
                return true;
            }
            tabHit = true;
            log.debug("Reject TAB: old focus owner is a matching bean comboBox JTextField editor, so have a hit!!!: " + oldFocusOwner);
            return false;
        }
        JTextField old = (JTextField) oldFocusOwner;
        if (!Objects.equals("ComboBox.textField", old.getName())) {
            // coming or going to other widget
            if (tabHit) {
                return true;
            }
            tabHit = true;
            log.debug("Reject TAB: old focus owner is a matching bean comboBox JTextField editor, so have a hit!!!: " + oldFocusOwner);
            return false;
        }
        return true;
    }

    private void onFocusChanged(PropertyChangeEvent event) {
        Object newValue = event.getNewValue();
        if (Objects.equals(newValue, ui.getCombobox().getEditor().getEditorComponent())) {
            enabled = true;
            tabHit = false;
            selectedIndex = ui.getCombobox().getSelectedIndex();
            log.debug(String.format("Acquire focus on %s, initial selected index: %d", ui.getName(), selectedIndex));
        } else {
            if (newValue != null) {
                oldFocusOwner = newValue;
                log.debug("New other focus owner: " + oldFocusOwner);
            }
            if (enabled) {
                log.debug("Lost focus on " + ui.getName());
                enabled = false;
                oldFocusOwner = ui.getCombobox().getEditor().getEditorComponent();
                log.debug("New other focus owner: " + oldFocusOwner);
            }
        }
    }

    /**
     * Editor for the Combobox of the UI - uses the decorator
     */
    class JAXXFilterableComboBoxEditor implements ComboBoxEditor {

        final ComboBoxEditor wrapped;
        Object oldItem;

        public JAXXFilterableComboBoxEditor(ComboBoxEditor wrapped) {
            this.wrapped = wrapped;
        }

        @Override
        public JTextComponent getEditorComponent() {
            return (JTextComponent) wrapped.getEditorComponent();
        }

        @Override
        public Object getItem() {
            JTextComponent editor = getEditorComponent();
            Object newValue = editor.getText();
            if (log.isDebugEnabled()) {
                log.debug("getItem " + newValue + " - " + (newValue != null ? newValue.getClass() : null));
            }

            if (oldItem != null && getBeanType().isInstance(oldItem)) {
                // The original value is not a string. Should return the value in it's
                // original type.
                if (Objects.equals(newValue, decorator.decorate(oldItem))) {
                    newValue = oldItem;
                }
            }
            if (log.isDebugEnabled()) {
                log.debug("getItem 2 " + newValue + " - " + (newValue != null ? newValue.getClass() : null));
            }
            return newValue;
        }

        @Override
        public void setItem(Object anObject) {
            if (log.isDebugEnabled()) {
                log.debug("setItem " + anObject + " - " + (anObject != null ? anObject.getClass() : null));
            }
            Object item = anObject;
            if (anObject != null) {
                if (getBeanType().isInstance(anObject)) {
                    item = decorator.decorate(anObject);
                    oldItem = anObject;
                }
                try {
                    wrapped.setItem(item);

                } catch (IllegalStateException e) {
                    // fail silently
                }
            }
        }

        @Override
        public void selectAll() {
            wrapped.selectAll();
        }

        @Override
        public void addActionListener(ActionListener l) {
            wrapped.addActionListener(l);
        }

        @Override
        public void removeActionListener(ActionListener l) {
            wrapped.removeActionListener(l);
        }
    }
}
