package org.nuiton.jaxx.widgets.datetime;

/*-
 * #%L
 * JAXX :: Widgets DateTime
 * %%
 * Copyright (C) 2008 - 2020 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.bean.JavaBean;
import io.ultreia.java4all.jaxx.widgets.BeanUIHandlerSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jdesktop.swingx.JXDatePicker;
import org.jdesktop.swingx.JXMonthView;
import org.nuiton.jaxx.widgets.BeanUIUtil;

import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.KeyStroke;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyEvent;
import java.text.ParseException;
import java.util.Date;
import java.util.Objects;

/**
 * Created on 02/09/2020.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 3.0
 */
public class BeanDateEditorHandler extends BeanUIHandlerSupport<BeanDateEditor> {
    private static final Logger log = LogManager.getLogger(BeanDateEditorHandler.class);

    @Override
    protected String getProperty(BeanDateEditor ui) {
        return ui.getProperty();
    }

    @Override
    protected void prepareInit(String property) {
        log.debug(String.format("%s - init BeanDateEditor", ui.getName()));
        if (property == null || property.isEmpty()) {
            ui.setProperty(ui.getName());
        }
        ui.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(KeyStroke.getKeyStroke(KeyEvent.VK_F5, 0), "none");
        JFormattedTextField formattedTextField = ui.getEditor();
        formattedTextField.setEditable(true);
        JXMonthView monthView = new JXMonthView();
        monthView.setLowerBound(new Date(0));
        monthView.setTraversable(true);
        monthView.setZoomable(true);
        ui.setMonthView(monthView);

        if (BeanUIUtil.isAutoSelectOnFocus(ui)) {
            BeanUIUtil.addAutoSelectOnFocus(formattedTextField);
            formattedTextField.addFocusListener(new FocusAdapter() {
                @Override
                public void focusLost(FocusEvent e) {

                    JFormattedTextField source = (JFormattedTextField) e.getSource();
                    if (source.isEditValid()) {
                        try {
                            ui.commitEdit();
                            log.debug("quit date editor, commit value: " + source.getValue());
                        } catch (ParseException e1) {
                            // l'édition est valide donc pas de problème ici
                        }
                    }
                    super.focusLost(e);
                }
            });
        }
    }

    @Override
    protected void prepareBindFromBean(String property, JavaBean bean) {
        bean.addPropertyChangeListener(property, e -> {
            Date oldValue = ui.getDate();
            Date newValue = (Date) e.getNewValue();
            if (!Objects.equals(oldValue, newValue)) {
                log.debug(String.format("%s - [%s] get new value from bean: %s", ui.getName(), property, newValue));
                ui.setDate(newValue);
            }
        });
    }

    @Override
    protected void prepareBindToBean(String property, JavaBean bean) {
        ui.addActionListener(e -> {
            JXDatePicker source = (JXDatePicker) e.getSource();
            Date newValue = source.getDate();
            if (bean == null) {
                return;
            }
            if (!Objects.equals(newValue, bean.get(property))) {
                log.debug(String.format("%s - [%s] set new value to bean: %s", ui.getName(), property, newValue));
                bean.set(property, newValue);
            }
        });
    }
}
