/*
 * #%L
 * JAXX :: Compiler
 * %%
 * Copyright (C) 2008 - 2023 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.jaxx.compiler.tasks;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.nuiton.jaxx.compiler.JAXXCompiler;
import org.nuiton.jaxx.compiler.JAXXCompilerFile;
import org.nuiton.jaxx.compiler.JAXXEngine;

import java.net.URL;

/**
 * First compile pass task to validate jaxx files and look after dependencies.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 2.0.2
 */
public class CompileFirstPassTask extends JAXXEngineTask {

    /** Logger */
    private static final Logger log = LogManager.getLogger(CompileFirstPassTask.class);

    /** Task name */
    public static final String TASK_NAME = "CompileFirstPass";

    public CompileFirstPassTask() {
        super(TASK_NAME);
    }

    @Override
    public boolean perform(JAXXEngine engine) throws Exception {
        boolean success = true;

        JAXXCompilerFile[] undone = engine.getFilesToCompile();

        while (undone.length > 0) {

            // start a new round of compilation

            for (JAXXCompilerFile jaxxFile : undone) {

                boolean successForFile = treatFile(engine, jaxxFile);

                if (!successForFile) {

                    // something is wrong...
                    success = false;
                }
            }

            // prepare next round
            undone = engine.getFilesToCompile();
        }
        return success;
    }

    protected boolean treatFile(JAXXEngine engine,
                                JAXXCompilerFile jaxxFile) throws Exception {

        boolean isVerbose = engine.isVerbose();

        if (isVerbose) {
            log.info("start " + jaxxFile.getClassName());
        }

        boolean success = true;

        JAXXCompiler compiler = engine.newCompiler(jaxxFile);
        addStartProfileTime(engine, compiler);

        compiler.detectIdentStyleSheetFile();

        compiler.compileFirstPass();

//        if (!compiler.isIdentCssFound()) {
//
//            // check if we can add ident css file
//
//            File cssFile = jaxxFile.getCssFile();
//
//            if (log.isDebugEnabled()) {
//                log.debug("test ident css file " + cssFile + " : " + isVerbose);
//            }
//            if (cssFile.exists()) {
//
//                if (isVerbose) {
//                    log.info("Auto import of css " + cssFile);
//                }
//                // ok add it
//                compiler.registerStyleSheetFile(cssFile.toURI().toURL(), false);
//            }
//        }

        URL commonCss = engine.getConfiguration().getCommonCss();
        if (commonCss != null) {
            if (isVerbose) {
                log.info("Import common css: " + commonCss);
            }
            compiler.registerStyleSheetFile(commonCss, false);
        }

        addEndProfileTime(engine, compiler);

        if (compiler.isFailed()) {
            success = false;
        }
        return success;
    }
}
